/**
 * ActorSuggestion.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.flow.tree;

import java.util.Enumeration;
import java.util.Vector;

import adams.core.DebugHelper;
import adams.core.Properties;
import adams.core.Utils;
import adams.env.ActorSuggestionDefinition;
import adams.env.Environment;
import adams.flow.core.AbstractActor;

/**
 * Class for suggesting actors when editing a flow, depending on the context.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 */
public class ActorSuggestion {

  /** whether to output some debug information. */
  private final static boolean DEBUG = DebugHelper.isDebugOn(ActorSuggestion.class);

  /** the name of the props file. */
  public final static String FILENAME = "ActorSuggestion.props";

  /** the key for the default actor. */
  public final static String KEY_DEFAULT = "Default";

  /** the properties with the rules. */
  protected Properties m_Properties;

  /** the default actor(s). */
  protected AbstractActor[] m_Defaults;

  /** the valid rules. */
  protected String[] m_Rules;

  /** the singleton. */
  protected static ActorSuggestion m_Singleton;

  /**
   * Initializes the object.
   */
  private ActorSuggestion() {
    super();
    initialize();
  }

  /**
   * Initializes the rules engine for proposing actors.
   */
  protected void initialize() {
    String[]				parts;
    Vector<AbstractActor>		actors;
    Vector<String>			rules;
    int					i;
    Enumeration<String>			names;
    String				name;
    String				rule;
    adams.parser.ActorSuggestion	suggestion;

    m_Properties = Environment.getInstance().read(ActorSuggestionDefinition.KEY);

    // get the default(s)
    parts  = m_Properties.getProperty(KEY_DEFAULT, "adams.flow.transformer.PassThrough").split(",");
    actors = new Vector<AbstractActor>();
    for (i = 0; i < parts.length; i++)  {
      try {
	actors.add((AbstractActor) Class.forName(parts[i]).newInstance());
      }
      catch (Exception e) {
	System.err.println("Failed to instantiate default actor '" + parts[i] + "':");
	e.printStackTrace();
      }
    }
    if (actors.size() == 0)
      actors.add(new adams.flow.transformer.PassThrough());
    m_Defaults = actors.toArray(new AbstractActor[actors.size()]);

    if (DEBUG)
      DebugHelper.debug(getClass(), "Defaults: " + Utils.arrayToString(m_Defaults, true));

    // get the rules
    rules      = new Vector<String>();
    names      = (Enumeration<String>) m_Properties.propertyNames();
    suggestion = new adams.parser.ActorSuggestion();
    suggestion.setParent(new adams.flow.control.Flow());
    suggestion.setPosition(0);
    suggestion.setActors(new AbstractActor[0]);
    while (names.hasMoreElements()) {
      name = names.nextElement();
      if (name.equals(KEY_DEFAULT))
	continue;
      rule = m_Properties.getProperty(name);
      try {
	suggestion.setExpression(rule);
	suggestion.evaluate();
	rules.add(rule);
      }
      catch (Exception e) {
	System.err.println("Invalid actor suggestion rule: " + rule);
	e.printStackTrace();
      }
    }
    m_Rules = rules.toArray(new String[rules.size()]);

    if (DEBUG)
      DebugHelper.debug(getClass(), "Rules: " + Utils.arrayToString(m_Rules));
  }

  /**
   * Returns the available default actors.
   *
   * @return		the default actors
   */
  public AbstractActor[] getDefaults() {
    return m_Defaults;
  }

  /**
   * Returns the suggested actors.
   *
   * @param parent	the parent of the actor to suggest
   * @param position	the position of the actor to insert in the actors
   * @param actors	the actors to insert the suggested actor in
   * @return		the suggested actors
   */
  public AbstractActor[] suggest(AbstractActor parent, int position, AbstractActor[] actors) {
    AbstractActor[]		result;
    Vector<AbstractActor>	suggestions;
    AbstractActor[]		suggested;
    int				i;

    suggestions = new Vector<AbstractActor>();

    try {
      suggested = adams.parser.ActorSuggestion.evaluate(m_Rules, parent, position, actors);
      for (i = 0; i < suggested.length; i++) {
	if (suggested[i] != null)
	  suggestions.add(suggested[i]);
      }
    }
    catch (Exception e) {
      System.err.println("Failed to suggest actors:");
      e.printStackTrace();
    }

    result = suggestions.toArray(new AbstractActor[suggestions.size()]);

    if (DEBUG)
      DebugHelper.debug(
	  getClass(),
	    "suggest: "
	  + "parent=" + parent.getClass().getName() + ", "
	  + "position=" + position + ", "
	  + "actors=" + Utils.arrayToString(actors, true) + "\n"
	  + "--> " + Utils.arrayToString(result, true));

    return result;
  }

  /**
   * Returns the singleton instance for suggesting actors.
   *
   * @return		the singleton
   */
  public static synchronized ActorSuggestion getSingleton() {
    if (m_Singleton == null)
      m_Singleton = new ActorSuggestion();

    return m_Singleton;
  }
}
