/**
 * Variables.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.io.Serializable;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map;
import java.util.Vector;

import adams.core.base.BaseRegExp;
import adams.event.VariableChangeEvent;
import adams.event.VariableChangeEvent.Type;
import adams.event.VariableChangeListener;

/**
 * A container for storing variables and their values. Values are
 * stored as string representations. A variable placeholder string is of the
 * following form: <br/>
 * <pre>
 * @{name}
 * </pre>
 * With "name" consisting of word characters: 0-9a-zA-Z_-
 * <p/>
 * Environment variables can be accessed by prefixing them with "env."
 * ({@link Variables#ENVIRONMENT_VARIABLE_PREFIX}) and system properties
 * by prefixing with "system." ({@link Variables#SYSTEM_PROPERTY_PREFIX}).
 * Examples: system.os.name, env.PATH
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4468 $
 * @see #ENVIRONMENT_VARIABLE_PREFIX
 * @see #SYSTEM_PROPERTY_PREFIX
 */
public class Variables
  implements Serializable, CleanUpHandler, CloneHandler<Variables> {

  /** for serialization. */
  private static final long serialVersionUID = -1654017967835758598L;

  /** the start of a variable. */
  public final static String START = "@{";

  /** the end of a variable. */
  public final static String END = "}";

  /** allowed characters. */
  public final static String CHARS = "abcdefghijklmnopqrstuvwxyz_ABCDEFGHIJKLMNOPQRSTUVWXYZ-0123456789";

  /** the prefix for environment variables. */
  public final static String ENVIRONMENT_VARIABLE_PREFIX = "env.";

  /** the prefix for system properties. */
  public final static String SYSTEM_PROPERTY_PREFIX = "system.";

  /** the variable &lt;-&gt; value relation. */
  protected Hashtable<String,String> m_Variables;

  /** the environment variables &lt;-&gt; value relation. */
  protected Hashtable<String,String> m_EnvironmentVariables;

  /** the system properties &lt;-&gt; value relation. */
  protected Hashtable<String,String> m_SystemProperties;

  /** the listeners. */
  protected WrapperHashSet<VariableChangeListener> m_VariableChangeListeners;

  /** the debugging level. */
  private final static int DEBUGLEVEL = DebugHelper.getDebugLevel(Variables.class);

  /**
   * Initializes the container.
   */
  public Variables() {
    super();

    m_Variables               = new Hashtable<String,String>();
    m_VariableChangeListeners = new WrapperHashSet<VariableChangeListener>();

    // environment variables
    m_EnvironmentVariables = new Hashtable<String,String>();
    Map<String,String> env = System.getenv();
    for (String key: env.keySet())
      m_EnvironmentVariables.put(ENVIRONMENT_VARIABLE_PREFIX + key, env.get(key));

    // system properties
    m_SystemProperties = new Hashtable<String,String>();
    java.util.Properties props = System.getProperties();
    Enumeration enm = props.propertyNames();
    while (enm.hasMoreElements()) {
      String key = (String) enm.nextElement();
      m_SystemProperties.put(SYSTEM_PROPERTY_PREFIX + key, props.getProperty(key));
    }
  }

  /**
   * Outputs some debugging information.
   *
   * @param msg		the message to output
   */
  private void debug(String msg) {
    DebugHelper.debug(Variables.class, msg);
  }

  /**
   * Removes all currently stored variables.
   */
  public void clear() {
    m_Variables.clear();
  }

  /**
   * Returns an enumeration over the names of the variables.
   *
   * @return		the names
   */
  public Enumeration<String> names() {
    return m_Variables.keys();
  }

  /**
   * Stores the value for the variable.
   *
   * @param name	the name (or placeholder string) of the variable
   * @param value	the value of the variable
   */
  public void set(String name, String value) {
    Type	type;
    String	strippedName;

    if (!isValidName(name))
      throw new IllegalArgumentException("Invalid variable name: " + name);

    strippedName = extractName(name);
    if (strippedName.startsWith(ENVIRONMENT_VARIABLE_PREFIX))
      return;
    if (strippedName.startsWith(SYSTEM_PROPERTY_PREFIX))
      return;

    if (has(strippedName))
      type = Type.MODIFIED;
    else
      type = Type.ADDED;

    if (DEBUGLEVEL > 0)
      debug("set: name=" + name + ", value=" + value);

    m_Variables.put(strippedName, value);
    notifyVariableChangeListeners(new VariableChangeEvent(this, type, name));
  }

  /**
   * Checks whether a variable is stored or not.
   *
   * @param name	the name (or placeholder string) of the variable
   * @return		true if the variable is stored
   */
  public boolean has(String name) {
    name = extractName(name);
    if (m_EnvironmentVariables.containsKey(name))
      return true;
    else if (m_SystemProperties.containsKey(name))
      return true;
    else
      return m_Variables.containsKey(name);
  }

  /**
   * Returns the stored value if present, otherwise null.
   *
   * @param name	the name (or placeholder string) of the variable
   * @return		the associated value or null if not found
   */
  public String get(String name) {
    return get(name, null);
  }

  /**
   * Removes the variable.
   *
   * @param name	the name (or placeholder string) of the variable
   * @return		the previously stored value
   */
  public String remove(String name) {
    String	result;

    result = m_Variables.remove(extractName(name));
    notifyVariableChangeListeners(new VariableChangeEvent(this, Type.REMOVED, name));

    if (DEBUGLEVEL > 0)
      debug("remove: name=" + name + ", value=" + result);

    return result;
  }

  /**
   * Removes variables that match a regular expressions.
   *
   * @param regExp	the regular expression to match the name of the variables against
   * @return		true if at least one was removed
   */
  public boolean remove(BaseRegExp regExp) {
    boolean		result;
    Vector<String>	names;

    result = false;

    names = new Vector<String>();
    synchronized(m_Variables) {
      // find matches
      for (String name: m_Variables.keySet()) {
	if (regExp.isMatch(name))
	  names.add(name);
      }

      // remove variables
      for (String name: names)
	remove(name);
    }

    return result;
  }

  /**
   * Returns the stored value if present, otherwise the default value.
   *
   * @param name	the name (or placeholder string) of the variable
   * @param defValue	the default value, in case the variable is not stored
   * @return		the associated value
   */
  public String get(String name, String defValue) {
    name = extractName(name);
    if (m_EnvironmentVariables.containsKey(name))
      return m_EnvironmentVariables.get(name);
    else if (m_SystemProperties.containsKey(name))
      return m_SystemProperties.get(name);
    else if (m_Variables.containsKey(name))
      return m_Variables.get(name);
    else
      return defValue;
  }

  /**
   * Returns the number of variables currently stored.
   *
   * @return		the number of variables
   */
  public int size() {
    return m_Variables.size();
  }

  /**
   * Adds the listener to the internal list.
   *
   * @param l		the listener to add
   */
  public void addVariableChangeListener(VariableChangeListener l) {
    m_VariableChangeListeners.add(l);
  }

  /**
   * Removes the listener from the internal list.
   *
   * @param l		the listener to remove
   */
  public void removeVariableChangeListener(VariableChangeListener l) {
    m_VariableChangeListeners.remove(l);
  }

  /**
   * Removes all listeners from the internal list.
   */
  public void removeVariableChangeListeners() {
    m_VariableChangeListeners.clear();
  }

  /**
   * Notifies all listeners with the specified event.
   *
   * @param e		the event to send
   */
  protected void notifyVariableChangeListeners(VariableChangeEvent e) {
    for (VariableChangeListener l: m_VariableChangeListeners)
      l.variableChanged(e);
  }

  /**
   * Extracts the name of a variable from the placeholder string, i.e., it
   * strips the "@{" and "}" from the string. Does nothing if already stripped.
   *
   * @param variable	the variable placeholder
   * @return		the extracted name
   * @see		#START
   * @see		#END
   */
  public static String extractName(String variable) {
    String	result;

    result = variable;

    if (result.startsWith(START) && result.endsWith(END))
      result = result.substring(2, result.length() - 1);

    return result;
  }

  /**
   * Surrounds the variable name with "@{" and "}", if necessary.
   *
   * @param name	the name to process
   * @return		the padded name
   * @see		#START
   * @see		#END
   */
  public static String padName(String name) {
    String	result;

    result = name;
    if (!(result.startsWith(START) && result.endsWith(END)))
      result = START + result + END;

    return result;
  }

  /**
   * Checks whether the string represents a variable placeholder, i.e., it
   * starts with "@{" and ends with "}".
   *
   * @param s		the string to check
   * @return		true if the string represents a variable placeholder
   * 			string
   */
  public static boolean isPlaceholder(String s) {
    boolean	result;

    result = (s.startsWith(START) && s.endsWith(END));
    if (result)
      result = isValidName(extractName(s));

    return result;
  }

  /**
   * Checks whether the string represents a valid name (without the "@{" and "}").
   *
   * @param s		the name to check
   * @return		true if valid
   */
  public static boolean isValidName(String s) {
    boolean	result;
    String	name;

    name   = extractName(s);
    result = (name.length() > 0);
    if (result) {
      name   = name.replaceAll("\\w", "").replace("-", "");
      result = (name.length() == 0);
    }

    return result;
  }

  /**
   * Replaces all variables in the string with the currently stored values.
   *
   * @param s		the string to process
   * @return		the processed string
   */
  public String expand(String s) {
    String		result;
    Enumeration<String>	enm;
    String		name;
    String		part;

    result = s;

    // environment variables
    part = Variables.START + ENVIRONMENT_VARIABLE_PREFIX;
    if (result.indexOf(part) > -1) {
      enm = m_EnvironmentVariables.keys();
      while (enm.hasMoreElements() && (result.indexOf(part) > -1)) {
	name   = enm.nextElement();
	result = result.replace(START + name + END, get(name));
      }
    }

    // system properties
    part = Variables.START + SYSTEM_PROPERTY_PREFIX;
    if (result.indexOf(part) > -1) {
      enm = m_SystemProperties.keys();
      while (enm.hasMoreElements() && (result.indexOf(part) > -1)) {
	name   = enm.nextElement();
	result = result.replace(START + name + END, get(name));
      }
    }

    // regular variables
    part = Variables.START;
    if (result.indexOf(part) > -1) {
      enm = names();
      while (enm.hasMoreElements() && (result.indexOf(part) > -1)) {
	name   = enm.nextElement();
	result = result.replace(START + name + END, get(name));
      }
    }

    return result;
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    removeVariableChangeListeners();
    clear();
  }

  /**
   * Returns a clone of the object (but without the listeners).
   *
   * @return		the clone
   */
  public Variables getClone() {
    Variables	result;

    result = new Variables();
    result.m_Variables = (Hashtable<String,String>) m_Variables.clone();

    return result;
  }

  /**
   * Simply returns the internal hashtable of variable/value pairs.
   *
   * @return		the stored variables
   */
  public String toString() {
    return m_Variables.toString();
  }
}
