/*
 * AbstractCSVReportWriter.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.data.io.output;

import java.util.Collections;
import java.util.Vector;

import adams.core.io.CsvSpreadSheetWriter;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;
import adams.data.io.input.AbstractSimpleCSVReportReader;
import adams.data.report.AbstractField;
import adams.data.report.DataType;
import adams.data.report.Report;

/**
 * Abstract ancestor for writing reports in CSV format.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 * @param <T> the type of report to handle
 */
public abstract class AbstractSimpleCSVReportWriter<T extends Report>
  extends AbstractReportWriter<T> {

  /** for serialization. */
  private static final long serialVersionUID = 1068874780353203514L;

  /** the name of the "field" column. */
  public final static String COL_FIELD = "Field";

  /** the name of the "type" column. */
  public final static String COL_TYPE = "Type";

  /** the name of the "value" column. */
  public final static String COL_VALUE = "Value";

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Writes reports in CSV file format.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "CSV file format";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension(s) (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{AbstractSimpleCSVReportReader.FILE_EXTENSION};
  }

  /**
   * Performs the actual writing.
   *
   * @param data	the data to write
   * @return		true if successfully written
   */
  protected boolean writeData(T data) {
    boolean			result;
    SpreadSheet			sheet;
    Vector<AbstractField>	fields;
    int				i;
    Row				row;

    sheet = new SpreadSheet();

    // header
    row = sheet.getHeaderRow();
    row.addCell(COL_FIELD).setContent(COL_FIELD);
    row.addCell(COL_TYPE).setContent(COL_TYPE);
    row.addCell(COL_VALUE).setContent(COL_VALUE);

    // the parent ID
    row = sheet.addRow(Report.PROPERTY_PARENTID);
    row.addCell(COL_FIELD).setContent(Report.PROPERTY_PARENTID);
    row.addCell(COL_TYPE).setContent(DataType.NUMERIC.toString());
    row.addCell(COL_VALUE).setContent("" + data.getDatabaseID());

    // transfer properties
    fields = data.getFields();
    Collections.sort(fields);
    for (i = 0; i < fields.size(); i++) {
      row = sheet.addRow(fields.get(i).toString());
      row.addCell(COL_FIELD).setContent(fields.get(i).toString());
      row.addCell(COL_TYPE).setContent(fields.get(i).getDataType().toString());
      row.addCell(COL_VALUE).setContent(data.getValue(fields.get(i)).toString());
    }

    // write CSV file
    result = new CsvSpreadSheetWriter().write(sheet, m_Output.getAbsolutePath());
    if (!result)
      getSystemErr().println("Error writing report #" + data.getDatabaseID() + " to " + m_Output);

    return result;
  }
}
