/*
 * DatabaseConnection.java
 * Copyright (C) 2008-2011 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.db;

import adams.core.base.BasePassword;
import adams.env.DatabaseConnectionDefinition;

/**
 * DatabaseConnection manages the interface to the database back-end.
 * Currently set up for MYSQL.
 *
 *  @author  dale (dale at waikato dot ac dot nz)
 *  @version $Revision: 3544 $
 */
public class DatabaseConnection
  extends AbstractDatabaseConnection {

  /** for serialization. */
  private static final long serialVersionUID = -3625820307854172417L;

  /** the props file. */
  public final static String FILENAME = "DatabaseConnection.props";

  /** for managing the database connections. */
  private static DatabaseManager<DatabaseConnection> m_DatabaseManager;
  static {
    m_DatabaseManager = new DatabaseManager<DatabaseConnection>("adams");
    DatabaseConnection dbcon = new DatabaseConnection();
    m_DatabaseManager.setDefault(DatabaseConnection.getSingleton(dbcon.getURL(), dbcon.getUser(), dbcon.getPassword()));
  }

  /**
   * Local Database Constructor.
   */
  public DatabaseConnection() {
    super();
  }

  /**
   * Local Database Constructor. Initialise the JDBC driver, and attempt
   * connection to the database specified in the URL, with the given username
   * and password.
   *
   * @param driver      the JDBC driver
   * @param url         the JDBC URL
   * @param user        the user to connect with
   * @param password    the password for the user
   */
  public DatabaseConnection(String url, String user, BasePassword password) {
    super(url, user, password);
  }

  /**
   * Returns the properties key to use for retrieving the properties.
   *
   * @return		the key
   */
  protected String getDefinitionKey() {
    return DatabaseConnectionDefinition.KEY;
  }

  /**
   * Returns the global database connection object. If not instantiated yet, it
   * will automatically try to connect to the database server.
   *
   * @param url		the database URL
   * @param user	the database user
   * @param password	the database password
   * @return		the singleton
   */
  public static synchronized DatabaseConnection getSingleton(String url, String user, BasePassword password) {
    if (!m_DatabaseManager.has(url, user)) {
      m_DatabaseManager.add(new DatabaseConnection(url, user, password));
    }
    else {
      if (!m_DatabaseManager.get(url, user).isConnected()) {
	try {
	  m_DatabaseManager.get(url, user).connect();
	}
	catch (Exception e) {
	  e.printStackTrace();
	}
      }
    }

    return m_DatabaseManager.get(url, user);
  }

  /**
   * Returns the global database connection object. If not instantiated yet, it
   * can automatically try to connect to the database server, depending on the
   * default in the props file (SUFFIX_CONNECTONSTARTUP).
   *
   * @return		the singleton
   * @see		#getConnectOnStartUp()
   * @see		AbstractDatabaseConnection#SUFFIX_CONNECTONSTARTUP
   */
  public static synchronized DatabaseConnection getSingleton() {
    return m_DatabaseManager.getDefault();
  }
}
