/*
 * SequencePlotterContainer.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.container;

import java.util.Enumeration;
import java.util.Vector;

/**
 * A container for a single plot item. Can be used to name plots.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3932 $
 */
public class SequencePlotterContainer
  extends AbstractContainer {

  /** for serialization. */
  private static final long serialVersionUID = 7610036927945350224L;

  /** the identifier for the PlotName. */
  public final static String VALUE_PLOTNAME = "PlotName";

  /** the identifier for the X value. */
  public final static String VALUE_X = "X";

  /** the identifier for the Y value. */
  public final static String VALUE_Y = "Y";

  /** the identifier for the "is marker" value. */
  public final static String VALUE_ISMARKER = "IsMarker";

  /** the default plot name. */
  public static final String DEFAULT_PLOTNAME = "Plot";

  /**
   * Initializes the container with the default plot name and no X value and Y
   * value of 0.0.
   * <p/>
   * Only used for generating help information.
   */
  public SequencePlotterContainer() {
    this(DEFAULT_PLOTNAME, 0.0);
  }

  /**
   * Initializes the container with the specified plot name and no X value.
   *
   * @param plotName	the name of the plot
   * @param y		the y value of the plot
   */
  public SequencePlotterContainer(String plotName, Double y) {
    this(plotName, y, false);
  }

  /**
   * Initializes the container with the specified plot name and no X value.
   *
   * @param plotName	the name of the plot
   * @param y		the y value of the plot
   * @param isMarker	whether this container represents a marker
   */
  public SequencePlotterContainer(String plotName, Double y, boolean isMarker) {
    this(plotName, null, y, isMarker);
  }

  /**
   * Initializes the container with the default plot name.
   *
   * @param x		the x value of the plot
   * @param y		the y value of the plot
   * @see		#DEFAULT_PLOTNAME
   */
  public SequencePlotterContainer(Double x, Double y) {
    this(DEFAULT_PLOTNAME, x, y);
  }

  /**
   * Initializes the container with the default plot name.
   *
   * @param plotName	the name of the plot
   * @param x		the x value of the plot
   * @param y		the y value of the plot
   */
  public SequencePlotterContainer(String plotName, Double x, Double y) {
    this(plotName, x, y, false);
  }

  /**
   * Initializes the container with the default plot name.
   *
   * @param plotName	the name of the plot
   * @param x		the x value of the plot
   * @param y		the y value of the plot
   * @param isMarker	whether this container represents a marker
   */
  public SequencePlotterContainer(String plotName, Double x, Double y, boolean isMarker) {
    super();

    store(VALUE_PLOTNAME, plotName);
    store(VALUE_X, x);
    store(VALUE_Y, y);
    store(VALUE_ISMARKER, isMarker);
  }

  /**
   * Returns all value names that can be used (theoretically).
   *
   * @return		enumeration over all possible value names
   */
  public Enumeration<String> names() {
    Vector<String>	result;

    result = new Vector<String>();

    result.add(VALUE_PLOTNAME);
    result.add(VALUE_X);
    result.add(VALUE_Y);
    result.add(VALUE_ISMARKER);

    return result.elements();
  }

  /**
   * Checks whether the setup of the container is valid.
   *
   * @return		true if all the necessary values are available
   */
  public boolean isValid() {
    return   (hasValue(VALUE_PLOTNAME) && hasValue(VALUE_Y) && !hasValue(VALUE_Y) && hasValue(VALUE_ISMARKER))
           | (hasValue(VALUE_PLOTNAME) && hasValue(VALUE_X) &&  hasValue(VALUE_Y) && hasValue(VALUE_ISMARKER));
  }
}
