/*
 * AbstractDataContainerFileReader.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Hashtable;
import java.util.Vector;

import adams.core.io.PlaceholderFile;
import adams.data.container.DataContainer;
import adams.data.io.input.AbstractDataContainerReader;
import adams.data.io.input.IncrementalDataContainerReader;
import adams.flow.core.Token;
import adams.flow.provenance.ActorType;
import adams.flow.provenance.Provenance;
import adams.flow.provenance.ProvenanceContainer;
import adams.flow.provenance.ProvenanceInformation;
import adams.flow.provenance.ProvenanceSupporter;

/**
 * Abstract ancestor for actors that read data containers from disk.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 * @param <T> the type of data that is read from disk
 */
public abstract class AbstractDataContainerFileReader<T extends DataContainer>
  extends AbstractTransformer
  implements ProvenanceSupporter {

  /** for serialization. */
  private static final long serialVersionUID = 2136481673137019370L;

  /** the key for storing the current containers in the backup. */
  public final static String BACKUP_CONTAINERS = "containers";

  /** the reader to use. */
  protected AbstractDataContainerReader<T> m_Reader;

  /** the chromatograms that were read. */
  protected Vector<T> m_Containers;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "reader", "reader",
	    getDefaultReader());
  }

  /**
   * Returns the default reader to use.
   *
   * @return		the default reader
   */
  protected abstract AbstractDataContainerReader getDefaultReader();

  /**
   * Sets the reader to use.
   *
   * @param value	the filter
   */
  public void setReader(AbstractDataContainerReader value) {
    m_Reader = value;
    reset();
  }

  /**
   * Returns the reader in use.
   *
   * @return		the reader
   */
  public AbstractDataContainerReader getReader() {
    return m_Reader;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String readerTipText() {
    return "The reader to use for importing the data.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("reader");

    if (variable != null)
      return variable;
    else if (m_Reader != null)
      return m_Reader.getClass().getName();
    else
      return null;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the data type
   */
  public abstract Class[] generates();

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_CONTAINERS);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_CONTAINERS, m_Containers);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_CONTAINERS)) {
      m_Containers = (Vector<T>) state.get(BACKUP_CONTAINERS);
      state.remove(BACKUP_CONTAINERS);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Containers = new Vector<T>();
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    PlaceholderFile	file;

    result = null;

    file = new PlaceholderFile((String) m_InputToken.getPayload());

    // setup reader
    m_Reader.setInput(file);
    if (isDebugOn())
      debug("Attempting to load '" + file + "'");

    // read data
    try {
      m_Containers = m_Reader.read();
      if (isDebugOn())
	debug(m_Containers.size() + " containers read");
      if (!(m_Reader instanceof IncrementalDataContainerReader))
	m_Reader.cleanUp();
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      m_Containers.clear();
      result = "Error reading '" + file + "': " + e;
      return result;
    }

    return result;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    // read more data?
    if (m_Reader instanceof IncrementalDataContainerReader) {
      if (m_Containers.size() == 0)
	m_Containers = m_Reader.read();
    }

    result = new Token(m_Containers.get(0));
    m_Containers.remove(0);

    updateProvenance(result);

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    if (m_Reader instanceof IncrementalDataContainerReader)
      return (m_Containers.size() > 0) || ((IncrementalDataContainerReader) m_Reader).hasMoreData();
    else
      return (m_Containers.size() > 0);
  }

  /**
   * Updates the provenance information in the provided container.
   *
   * @param cont	the provenance container to update
   */
  public void updateProvenance(ProvenanceContainer cont) {
    if (Provenance.getSingleton().isEnabled())
      cont.addProvenance(new ProvenanceInformation(ActorType.DATAGENERATOR, this, ((Token) cont).getPayload().getClass()));
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    super.cleanUp();

    m_Reader.cleanUp();
  }
}
