/*
 * ArrayCombinations.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;


import java.util.Hashtable;
import java.util.Vector;

import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Turns an array of any type into a sequence of array combinations of given size.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown[]<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ArrayCombinations
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-length &lt;int&gt; (property: length)
 * &nbsp;&nbsp;&nbsp;the r in nCr
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-subsets &lt;COMBINATIONS|PERMUTATIONS&gt; (property: subsets)
 * &nbsp;&nbsp;&nbsp;combinations or permutations.
 * &nbsp;&nbsp;&nbsp;default: COMBINATIONS
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  Dale (dale at cs dot waikato dot ac dot nz)
 * @version $Revision: 3155 $
 */
public class ArrayCombinations
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -1405432778848290110L;

  /** the key for storing the current counter in the backup. */
  public final static String BACKUP_ELEMENTS = "elements";

  /** number of elements in subarray, */
  protected int m_Length;

  /** the remaining elements of the array that still need to be broadcasted. */
  protected Vector m_Elements;

  /** the subset type. */
  protected SubsetsType m_Subsets;

  /**
   * Defines whether to do combinations or permutations.
   *
   * @author  dale (dale at waikato dot ac dot nz)
   * @version $Revision: 3155 $
   */
  public enum SubsetsType {
    /** combinations. */
    COMBINATIONS,
    /** permutations. */
    PERMUTATIONS
  }

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Turns an array of any type into a sequence of array combinations of given size.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "length", "length",
	    1, 1, null);

    m_OptionManager.add(
	    "subsets", "subsets",
	    SubsetsType.COMBINATIONS);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("subsets");

    if (variable != null)
      return variable;
    else
      return m_Subsets.toString();
  }

  /**
   * Choose combinations or permutations.
   *
   * @param value	the action
   */
  public void setSubsets(SubsetsType value) {
    m_Subsets = value;
    reset();
  }

  /**
   * combinations or permutations.
   *
   * @return		the action
   */
  public SubsetsType getSubsets() {
    return m_Subsets;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String subsetsTipText() {
    return "combinations or permutations.";
  }

  /**
   * Set length.
   *
   * @param len	 length
   */
  public void setLength(int len){
    m_Length=len;
    reset();
  }

  /**
   * Get length.
   *
   * @return	length
   */
  public int getLength(){
    return(m_Length);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String lengthTipText(){
    return("the r in nCr");
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_ELEMENTS);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_ELEMENTS, m_Elements);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_ELEMENTS)) {
      m_Elements = (Vector) state.get(BACKUP_ELEMENTS);
      state.remove(BACKUP_ELEMENTS);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Elements = new Vector();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown[].class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result = new Token(m_Elements.get(0));
    m_Elements.remove(0);

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_Elements.size() > 0);
  }

  protected Object[] remove(Object[] obj, int pos){
    Object ret[] =new Object[obj.length-1];
    int count=0;
    for (int i=0;i<obj.length;i++){
      if (i != pos){
	ret[count++]=obj[i];
      }
    }
    return(ret);
  }

  protected Object[] removeUpToIncluding(Object[] obj, int pos){
    Object ret[] =new Object[obj.length-(pos+1)];
    int count=0;
    for (int i=0;i<obj.length;i++){
      if (i > pos){
	ret[count++]=obj[i];
      }
    }
    return(ret);
  }

  protected Object[] combine(Object o, Object arr[]){
    Object ret[] =new Object[arr.length+1];
    ret[0]=o;
    for (int i=0;i<arr.length;i++){
      ret[i+1]=arr[i];
    }
    return(ret);
  }

  protected Vector<Object[]> genCombinations(Object[] in, int num){
    Vector<Object[]> vobj=new Vector<Object[]>();
    if (num == 0){
      return(vobj);
    }
    for (int i=0;i<in.length;i++){
      // if unable to complete, continue
      if (in.length < num){
	continue;
      }
      Vector<Object[]> combs;
      if (num == 1){
	Object[] o=new Object[1];
	o[0]=in[i];
	vobj.add(o);
      } else {
	if (m_Subsets == SubsetsType.COMBINATIONS){
	  combs=genCombinations(removeUpToIncluding(in,i),num-1); // combinations
	} else {
	  combs=genCombinations(remove(in,i),num-1); // permutations
	}
	for (Object[] arr:combs){
	  vobj.add(combine(in[i],arr));
	}
      }

    }
    return(vobj);
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Object[]	array;

    result = null;

    try {
      m_Elements.clear();
      array = (Object[]) m_InputToken.getPayload();
      Vector<Object[]> vobj=genCombinations(array,m_Length);
      for (Object[] arr:vobj){
	m_Elements.add(arr);
      }
      //for (i = 0; i < array.length; i++)
	//m_Elements.add(array[i]);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
