/**
 * AbstractEditorRegistration.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.goe;

import java.io.Serializable;

import adams.core.ClassLister;

/**
 * Ancestor for classes that register GenericObjectEditor editors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3956 $
 */
public abstract class AbstractEditorRegistration
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = 4364270584642868600L;

  /**
   * Returns whether registration already occurred.
   *
   * @return		true if registration already occurred
   */
  protected abstract boolean hasRegistered();

  /**
   * Performs the registration of the editors.
   *
   * @return		true if registration successful
   */
  protected abstract boolean doRegister();

  /**
   * Performs the registration of the editors.
   *
   * @return		true if registration successful
   */
  public boolean register() {
    if (hasRegistered())
      return true;

    return doRegister();
  }

  /**
   * Returns a list with classnames of registration classes.
   *
   * @return		the registration classnames
   */
  public static String[] getRegistrations() {
    return ClassLister.getSingleton().getClassnames(AbstractEditorRegistration.class);
  }

  /**
   * Registers the GOE editors, using all available registration schemes.
   */
  public static synchronized void registerEditors() {
    String[]			registrations;
    AbstractEditorRegistration	registration;

    registrations = getRegistrations();
    for (String classname: registrations) {
      try {
	registration = (AbstractEditorRegistration) Class.forName(classname).newInstance();
	if (registration instanceof AdamsEditorsRegistration)
	  continue;
	if (!registration.register())
	  System.err.println("Failed to register editors successfully: " + classname);
      }
      catch (Exception e) {
	System.err.println("Failed to register editors: " + classname);
	e.printStackTrace();
      }
    }

    // register ADAMS as last one
    registration = new AdamsEditorsRegistration();
    if (!registration.register())
      System.err.println("Failed to register ADMAS editors successfully!");
  }
}
