/**
 * BaseRegExpEditor.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.goe;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Arrays;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import adams.core.Utils;
import adams.core.base.BaseObject;
import adams.core.base.BaseRegExp;
import adams.core.option.AbstractOption;
import adams.gui.core.BrowserHelper;
import adams.gui.core.GUIHelper;
import adams.gui.dialog.AbstractApprovalDialog;

/**
 * Editor specifically designed for entering regular expression. In order to
 * enter line feeds, tabs, etc. the user can enable "escaped input".
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4477 $
 */
public class BaseRegExpEditor
  extends BaseObjectEditor
  implements MultiSelectionEditor {

  /** the URL for the pattern class to use. */
  public final static String URL_REGEXP = "http://download.oracle.com/javase/1.6.0/docs/api/java/util/regex/Pattern.html";

  /** the checkbox for "escaped input". */
  protected JCheckBox m_CheckBoxEscapedInput;

  /** the help button for bringing up a browser with the Java Pattern class. */
  protected JButton m_ButtonHelp;

  /**
   * Returns the Compound as string.
   *
   * @param option	the current option
   * @param object	the Compound object to convert
   * @return		the generated string
   */
  public static String toString(AbstractOption option, Object object) {
    return ((BaseRegExp) object).stringValue();
  }

  /**
   * Returns a Compound generated from the string.
   *
   * @param option	the current option
   * @param str		the string to convert to a Compound
   * @return		the generated Compound
   */
  public static Object valueOf(AbstractOption option, String str) {
    return new BaseRegExp(Utils.unbackQuoteChars(str));
  }

  /**
   * Gets the custom editor component.
   *
   * @return 		the editor
   */
  protected JComponent createCustomEditor() {
    JPanel	panelAll;
    JPanel	panelCheck;
    JPanel	panel;
    JLabel	label;
    JPanel 	panelButtons;
    JButton 	buttonOK;
    JButton 	buttonClose;

    panelAll = new JPanel(new BorderLayout());
    panel    = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panelAll.add(panel, BorderLayout.NORTH);
    m_TextValue = new JTextField(20);
    m_TextValue.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent e) {
	if (e.getKeyCode() == KeyEvent.VK_ENTER) {
	  e.consume();
	  acceptInput();
	}
	else if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
	  e.consume();
	  discardInput();
	}
	else {
	  super.keyPressed(e);
	}
      }
    });
    label = new JLabel("Value");
    label.setDisplayedMnemonic('V');
    label.setLabelFor(m_TextValue);
    m_ButtonHelp = new JButton(GUIHelper.getIcon("help2.png"));
    m_ButtonHelp.setToolTipText("Opens up browser with help regarding regular expressions");
    m_ButtonHelp.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	BrowserHelper.openURL(URL_REGEXP);
      }
    });
    panel.add(label);
    panel.add(m_TextValue);
    panel.add(m_ButtonHelp);

    panelCheck = new JPanel(new BorderLayout());
    panelAll.add(panelCheck, BorderLayout.CENTER);
    panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panelCheck.add(panel, BorderLayout.NORTH);
    m_CheckBoxEscapedInput = new JCheckBox("Escaped input");
    m_CheckBoxEscapedInput.setSelected(true);
    m_CheckBoxEscapedInput.setMnemonic('E');
    m_CheckBoxEscapedInput.setToolTipText("If checked, you can eg enter new lines like '\\n' or tabs like '\\t' (without the quotes)");
    m_CheckBoxEscapedInput.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	if (m_CheckBoxEscapedInput.isSelected())
	  m_TextValue.setText(Utils.backQuoteChars(m_TextValue.getText()));
	else
	  m_TextValue.setText(Utils.unbackQuoteChars(m_TextValue.getText()));
      }
    });
    panelCheck.add(m_CheckBoxEscapedInput);

    panelButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    panelAll.add(panelButtons, BorderLayout.SOUTH);

    buttonOK = new JButton("OK");
    buttonOK.setMnemonic('O');
    buttonOK.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	acceptInput();
      }
    });
    panelButtons.add(buttonOK);

    buttonClose = new JButton("Cancel");
    buttonClose.setMnemonic('C');
    buttonClose.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	discardInput();
      }
    });
    panelButtons.add(buttonClose);

    return panelAll;
  }

  /**
   * Unescapes the string if necessary.
   *
   * @param s		the string to unescape
   * @return		the processed string
   */
  protected String unEscape(String s) {
    if (m_CheckBoxEscapedInput.isSelected())
      return Utils.unbackQuoteChars(s);
    else
      return s;
  }

  /**
   * Checks whether the string is valid.
   *
   * @param s		the string to check
   * @return		true if the string is valid
   */
  protected boolean isValid(String s) {
    return super.isValid(unEscape(s));
  }

  /**
   * Checks whether the string is the same as the currently used one.
   *
   * @param s		the string to check
   * @return		true if the strings are the same
   */
  protected boolean isUnchanged(String s) {
    return super.isUnchanged(unEscape(s));
  }

  /**
   * Parses the given string and returns the generated object. The string
   * has to be a valid one, i.e., the isValid(String) check has been
   * performed already and succeeded.
   *
   * @param s		the string to parse
   * @return		the generated object, or null in case of an error
   */
  protected BaseObject parse(String s) {
    return super.parse(unEscape(s));
  }

  /**
   * Initializes the display of the value.
   */
  protected void initForDisplay() {
    String	value;

    if (m_CheckBoxEscapedInput.isSelected())
      value = Utils.backQuoteChars(((BaseRegExp) getValue()).getValue());
    else
      value = ((BaseRegExp) getValue()).getValue();

    if (!m_TextValue.getText().equals(value))
      m_TextValue.setText(value);
    m_TextValue.setToolTipText(((BaseObject) getValue()).getTipText());
    m_TextValue.grabFocus();
  }

  /**
   * Returns the string to paint.
   *
   * @return		the string
   * @see		#paintValue(Graphics, Rectangle)
   */
  protected String getStringToPaint() {
    return ((BaseRegExp) getValue()).stringValue();
  }

  /**
   * Returns a custom string representation of the object.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toCustomStringRepresentation(Object obj) {
    return toString(null, obj);
  }

  /**
   * Returns an object created from the custom string representation.
   *
   * @param str		the string to turn into an object
   * @return		the object
   */
  public Object fromCustomStringRepresentation(String str) {
    return valueOf(null, str);
  }

  /**
   * Returns the selected objects.
   *
   * @return		the objects
   */
  public Object[] getSelectedObjects() {
    BaseRegExp[]		result;
    MultiLineValueDialog	dialog;
    Component			comp;
    Vector<String>		lines;
    int				i;

    // we need the parent
    comp = getCustomEditor();
    initForDisplay();

    dialog = new MultiLineValueDialog();
    dialog.setInfoText("Enter the regular expressions, one per line:");
    dialog.setLocationRelativeTo(comp.getParent());
    dialog.setVisible(true);

    if (dialog.getOption() == AbstractApprovalDialog.APPROVE_OPTION) {
      lines = new Vector<String>(Arrays.asList(dialog.getContent().split("\n")));
      Utils.removeEmptyLines(lines);
      result = new BaseRegExp[lines.size()];
      for (i = 0; i < lines.size(); i++)
	result[i] = (BaseRegExp) parse(lines.get(i));
    }
    else {
      result = new BaseRegExp[0];
    }

    return result;
  }
}
