/*
 * AbstractDateBasedAxisModel.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core.axis;

/**
 * Ancestor axis model for displaying date-based values.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3843 $
 */
public abstract class AbstractDateBasedAxisModel
  extends AbstractAxisModel {

  /** for serialization. */
  private static final long serialVersionUID = 6882846237550109166L;

  /**
   * Initializes the member variables.
   */
  protected void initialize() {
    super.initialize();

    m_Formatter = Formatter.getDateFormatter(getDefaultDateFormat());
  }

  /**
   * Checks whether the data range can be handled by the model.
   *
   * @param min		the minimum value
   * @param max		the maximum value
   * @return		always true
   */
  public boolean canHandle(double min, double max) {
    return true;
  }

  /**
   * Returns the default format for the date/time formatter.
   *
   * @return		the format string
   */
  protected abstract String getDefaultDateFormat();

  /**
   * Returns the display name of this model.
   *
   * @return		the display name
   */
  public abstract String getDisplayName();

  /**
   * Returns the display string of the value for the tooltip, for instance.
   *
   * @param value	the value to turn into string
   * @return		the display string
   */
  public String valueToDisplay(double value) {
    return getActualFormatter().format(value);
  }

  /**
   * Returns the value from the display string.
   *
   * @param display	the string to turn into a value
   * @return		the value
   */
  public double displayToValue(String display) {
    return getActualFormatter().parse(display);
  }

  /**
   * Returns the position on the axis for the given value.
   *
   * @param value	the value to get the position for
   * @return		the corresponding position
   */
  public int valueToPos(double value) {
    int	result;
    int	size;

    validate();

    size   = getParent().getLength() - 1;
    result = (int) Math.round((value - m_ActualMinimum) / (m_ActualMaximum - m_ActualMinimum) * size);

    return result;
  }

  /**
   * Returns the value for the given position on the axis.
   *
   * @param pos	the position to get the corresponding value for
   * @return		the corresponding value
   */
  public double posToValue(int pos) {
    double	result;
    int	size;

    validate();

    size   = getParent().getLength() - 1;
    result = ((double) pos / (double) size) * (m_ActualMaximum - m_ActualMinimum) + m_ActualMinimum;

    return result;
  }
}