/**
 * DateUtils.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * A helper class for common Date-related operations.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4466 $
 */
public class DateUtils {

  /**
   * Returns a date formatting object, initialized with timestamp format.
   * 
   * @return		the formatter
   * @see		Constants#TIMESTAMP_FORMAT
   */
  public static DateFormat getTimestampFormatter() {
    return new DateFormat(Constants.TIMESTAMP_FORMAT);
  }

  /**
   * Checks whether the supplied parse pattern is valid.
   * 
   * @param pattern	the pattern to check
   * @return		true if valid
   */
  public static boolean isValid(String pattern) {
    try {
      new SimpleDateFormat(pattern);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }
  
  /**
   * Returns a date earlier relative to the provided one, e.g.,
   * earlier by 2 hours: earlier(date, Calendar.HOUR, 2).
   * 
   * @param date	the date to use as basis
   * @param type	the type of decrement, e.g., Calendar.HOUR
   * @param decrement	the amount to decrement the date with
   * @return		the new date
   * @see		Calendar
   */
  public static Date earlier(Date date, int type, int decrement) {
    Calendar	cal;
    
    cal = new GregorianCalendar();
    cal.setTime(date);
    cal.add(type, -Math.abs(decrement));
    
    return cal.getTime();
  }
  
  /**
   * Returns a date later relative to the provided one, e.g.,
   * later by 2 hours: later(date, Calendar.HOUR, 2).
   * 
   * @param date	the date to use as basis
   * @param type	the type of increment, e.g., Calendar.HOUR
   * @param increment	the amount to increment the date with
   * @return		the new date
   * @see		Calendar
   */
  public static Date later(Date date, int type, int increment) {
    Calendar	cal;
    
    cal = new GregorianCalendar();
    cal.setTime(date);
    cal.add(type, Math.abs(increment));
    
    return cal.getTime();
  }
  
  /**
   * Returns a date with a value changed, e.g., the hours set to a specific
   * value.
   * 
   * @param date	the date to use as basis
   * @param type	the type of value to change, e.g., Calendar.HOUR
   * @param value	the value to set
   * @return		the new date, null in case of an error
   * @see		Calendar
   */
  public static Date set(Date date, int type, int value) {
    Date	result;
    Calendar	cal;
    
    cal = new GregorianCalendar();
    cal.setTime(date);
    
    try {
      cal.set(type, value);
      result = cal.getTime();
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }
    
    return result;
  }
  
  /**
   * Returns whether the "check" date is before the "base" date.
   * 
   * @param base	the basis of the comparison
   * @param toCheck	the date to compare with
   * @return		true if "check" is before "base"
   */
  public static boolean isBefore(Date base, Date check) {
    Calendar	cal;
    
    cal = new GregorianCalendar();
    cal.setTime(check);
    
    return cal.before(base);
  }
  
  /**
   * Returns whether the "check" date is after the "base" date.
   * 
   * @param base	the basis of the comparison
   * @param toCheck	the date to compare with
   * @return		true if "check" is after "base"
   */
  public static boolean isAfter(Date base, Date check) {
    Calendar	cal;
    
    cal = new GregorianCalendar();
    cal.setTime(base);
    
    return cal.before(check);
  }
}
