/**
 * MovingAverage.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.data.conversion;

import java.util.Vector;

import adams.data.statistics.StatUtils;

/**
 <!-- globalinfo-start -->
 * Computes the average on a window of numbers that have passed through and outputs the average.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-window-size &lt;int&gt; (property: windowSize)
 * &nbsp;&nbsp;&nbsp;The number of data points to use for computing the average.
 * &nbsp;&nbsp;&nbsp;default: 10
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3937 $
 */
public class MovingAverage
  extends AbstractConversion
  implements StreamConversion {

  /** for serialization. */
  private static final long serialVersionUID = -4092302172529978800L;

  /** the size of the window to compute the average on. */
  protected int m_WindowSize;

  /** the values to compute the average on. */
  protected Vector<Number> m_Queue;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Computes the average on a window of numbers that have passed through "
      + "and outputs the average.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "window-size", "windowSize",
	    10, 1, null);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Queue = new Vector<Number>();
  }

  /**
   * Resets the members.
   */
  protected void reset() {
    super.reset();

    m_Queue.clear();
  }

  /**
   * Sets the window size.
   *
   * @param value	the window size
   */
  public void setWindowSize(int value) {
    if (value > 0) {
      m_WindowSize = value;
      reset();
    }
    else {
      getSystemErr().println("Minimum window size is 1, provided: " + value);
    }
  }

  /**
   * Returns the window size.
   *
   * @return 		the window zie
   */
  public int getWindowSize() {
    return m_WindowSize;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String windowSizeTipText() {
    return "The number of data points to use for computing the average.";
  }

  /**
   * Returns the class that is accepted as input.
   *
   * @return		the class
   */
  public Class accepts() {
    return Number.class;
  }

  /**
   * Returns the class that is generated as output.
   *
   * @return		the class
   */
  public Class generates() {
    return Double.class;
  }

  /**
   * Performs the actual conversion.
   *
   * @return		the converted data
   * @throws Exception	if something goes wrong with the conversion
   */
  protected Object doConvert() throws Exception {
    Double	result;

    result = null;

    m_Queue.add((Number) m_Input);
    while (m_Queue.size() > m_WindowSize)
      m_Queue.remove(0);

    if (m_Queue.size() == m_WindowSize)
      result = StatUtils.mean(m_Queue.toArray(new Number[m_Queue.size()]));

    return result;
  }
}
