/**
 * Cast.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.control;

import java.lang.reflect.Array;

import adams.flow.core.ControlActor;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.flow.transformer.AbstractTransformer;

/**
 <!-- globalinfo-start -->
 * Casts the incoming data to the specified classname. For arrays, use '[]' at the end.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Object<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Cast
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-classname &lt;java.lang.String&gt; (property: classname)
 * &nbsp;&nbsp;&nbsp;The classname to cast the incoming data to.
 * &nbsp;&nbsp;&nbsp;default: java.lang.Object
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4564 $
 */
public class Cast
  extends AbstractTransformer
  implements ControlActor {

  /** for serialization. */
  private static final long serialVersionUID = -6615127883045169960L;

  /** the Class to cast to. */
  protected String m_Classname;

  /** the actual class to cast to. */
  protected Class m_ActualClass;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
      "Casts the incoming data to the specified classname. For arrays, "
    + "use '[]' at the end.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "classname", "classname",
	    "java.lang.Object");
  }

  /**
   * Resets the actor.
   */
  protected void reset() {
    super.reset();

    m_ActualClass = null;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("classname");

    if (variable != null)
      return variable;
    else
      return m_Classname;
  }

  /**
   * Instantiates the class represented by the classname.
   *
   * @param classname	the class, [] define arrays
   */
  protected Class toClass(String classname) {
    Class	result;
    boolean	array;

    array = classname.endsWith("[]");
    if (array)
      classname = classname.substring(classname.length() - 2);

    try {
      if (array)
	result = Array.newInstance(Class.forName(classname), 0).getClass();
      else
	result = Class.forName(classname);
    }
    catch (Exception e) {
      result = null;
    }

    return result;
  }

  /**
   * Sets the classname.
   *
   * @param value	the classname
   */
  public void setClassname(String value) {
    if (toClass(value) != null) {
      m_Classname = value;
      reset();
    }
    else {
      getSystemErr().println("Failed to instantiate class: " + value);
    }
  }

  /**
   * Returns the classname.
   *
   * @return		the classname
   */
  public String getClassname() {
    return m_Classname;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String classnameTipText() {
    return "The classname to cast the incoming data to.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Object.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    Class	cls;

    cls = toClass(getClassname());

    if (cls == null)
      return new Class[]{Unknown.class};
    else
      return new Class[]{cls};
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      m_ActualClass = toClass(getClassname());
      if (m_ActualClass == null)
	result = "Failed to instantiate class '" + getClassname() + "'!";
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    String	msg;

    result = null;

    try {
      m_OutputToken = new Token(m_ActualClass.cast(m_InputToken.getPayload()));
    }
    catch (Exception e) {
      msg    = "Failed to cast to '" + getClassname() + "': ";
      result = msg + e;
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
    }

    return result;
  }
}
