/*
 * Injector.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.Hashtable;
import java.util.Vector;

import adams.flow.core.ControlActor;
import adams.flow.core.Token;
import adams.flow.transformer.AbstractTransformer;

/**
 <!-- globalinfo-start -->
 * Injects a string token into the token sequence.<br/>
 * The string can be inject before or after the current token. One can also control how often the string gets injected, i.e., every n-th token.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Injector
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-injection &lt;java.lang.String&gt; (property: injection)
 * &nbsp;&nbsp;&nbsp;The string to inject into the sequence.
 * &nbsp;&nbsp;&nbsp;default: inject_me
 * </pre>
 *
 * <pre>-location &lt;BEFORE|AFTER&gt; (property: location)
 * &nbsp;&nbsp;&nbsp;The location where to inject the string.
 * &nbsp;&nbsp;&nbsp;default: AFTER
 * </pre>
 *
 * <pre>-nth &lt;int&gt; (property: everyNth)
 * &nbsp;&nbsp;&nbsp;The number of tokens after which the injection takes place.
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2842 $
 */
public class Injector
  extends AbstractTransformer
  implements ControlActor {

  /** for serialization. */
  private static final long serialVersionUID = 5477999576142658922L;

  /** the key for storing the current counter in the backup. */
  public final static String BACKUP_COUNTER = "counter";

  /** the key for storing the queue in the backup. */
  public final static String BACKUP_QUEUE = "queue";

  /**
   * Enumeration for where to inject the String.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 2842 $
   */
  public enum Location {
    /** before the current token. */
    BEFORE,
    /** after the current token. */
    AFTER
  }

  /** the string to inject. */
  protected String m_Injection;

  /** where to inject the string. */
  protected Location m_Location;

  /** every nth token the string will get injected. */
  protected int m_EveryNth;

  /** the strings to output. */
  protected Vector<String> m_Queue;

  /** the token counter. */
  protected int m_Counter;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Injects a string token into the token sequence.\n"
      + "The string can be inject before or after the current token. One can "
      + "also control how often the string gets injected, i.e., every n-th token.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "injection", "injection",
	    "inject_me");

    m_OptionManager.add(
	    "location", "location",
	    Location.AFTER);

    m_OptionManager.add(
	    "nth", "everyNth",
	    1, 1, null);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("injection");

    if (variable != null)
      return variable;
    else if (m_Injection.length() > 0)
      return m_Injection;
    else
      return null;
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Queue = new Vector<String>();
  }

  /**
   * Sets the string to inject.
   *
   * @param value	the string
   */
  public void setInjection(String value) {
    m_Injection = value;
    reset();
  }

  /**
   * Returns the string to inject.
   *
   * @return		the string
   */
  public String getInjection() {
    return m_Injection;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String injectionTipText() {
    return "The string to inject into the sequence.";
  }

  /**
   * Sets the location where to inject the string.
   *
   * @param value	the location
   */
  public void setLocation(Location value) {
    m_Location = value;
    reset();
  }

  /**
   * Returns the location wher to inject the string.
   *
   * @return		the location
   */
  public Location getLocation() {
    return m_Location;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String locationTipText() {
    return "The location where to inject the string.";
  }

  /**
   * Sets after how many tokens the injection takes place.
   *
   * @param value	the number of tokens
   */
  public void setEveryNth(int value) {
    if (value >= 1) {
      m_EveryNth = value;
      reset();
    }
    else {
      getSystemErr().println(
	  "Value for every nth has to be at least 1, provided: " + value);
    }
  }

  /**
   * Returns after how many tokens the injection takes place.
   *
   * @return		the number of tokens
   */
 public int getEveryNth() {
    return m_EveryNth;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String everyNthTipText() {
    return "The number of tokens after which the injection takes place.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_COUNTER);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_COUNTER, m_Counter);
    result.put(BACKUP_QUEUE, m_Queue);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_COUNTER)) {
      m_Counter = (Integer) state.get(BACKUP_COUNTER);
      state.remove(BACKUP_COUNTER);
    }

    if (state.containsKey(BACKUP_QUEUE)) {
      m_Queue = (Vector<String>) state.get(BACKUP_QUEUE);
      state.remove(BACKUP_QUEUE);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Counter = 0;
    m_Queue.clear();
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;

    result = null;

    m_Counter++;

    m_Queue.add((String) m_InputToken.getPayload());

    if (m_Counter % m_EveryNth == 0) {
      if (m_Location == Location.BEFORE)
	m_Queue.insertElementAt(m_Injection, 0);
      else if (m_Location == Location.AFTER)
	m_Queue.addElement(m_Injection);
      else
	result = "Unhandled location: " + m_Location;
    }

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_Queue.size() > 0);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = new Token(m_Queue.get(0));
    m_Queue.remove(0);

    m_OutputToken = null;
    m_InputToken  = null;

    return result;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    m_Queue.clear();

    super.wrapUp();
  }
}
