/*
 * Sinkhole.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.Hashtable;

import adams.flow.condition.bool.AbstractBooleanCondition;
import adams.flow.condition.bool.Expression;
import adams.flow.core.ControlActor;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.flow.transformer.AbstractTransformer;
import adams.parser.BooleanExpressionText;
import adams.parser.GrammarSupplier;

/**
 <!-- globalinfo-start -->
 * Blocks for propagation of tokens if the condition evaluates to 'true'.In case of integer or double tokens that arrive at the input, these can be accessed in the expression via 'X'.<br/>
 * <br/>
 * The following grammar is used for evaluating the boolean expressions (depends on the selected condition):<br/>
 * <br/>
 *  expr_list ::= expr_list expr_part | expr_part ;<br/>
 *  expr_part ::= boolexpr ;<br/>
 * <br/>
 *  boolexpr ::=    ( boolexpr )<br/>
 *                | true<br/>
 *                | false<br/>
 *                | numexpr &lt; numexpr<br/>
 *                | numexpr &lt;= numexpr<br/>
 *                | numexpr &gt; numexpr<br/>
 *                | numexpr &gt;= numexpr<br/>
 *                | numexpr = numexpr<br/>
 *                | numexpr != numexpr<br/>
 *                | strexpr &lt; strexpr<br/>
 *                | strexpr &lt;= strexpr<br/>
 *                | strexpr &gt; strexpr<br/>
 *                | strexpr &gt;= strexpr<br/>
 *                | strexpr = strexpr<br/>
 *                | strexpr != strexpr<br/>
 *                | matches ( strexpr , regexp )<br/>
 *                | ! boolexpr<br/>
 *                | boolexpr &amp; boolexpr<br/>
 *                | boolexpr | boolexpr<br/>
 *                ;<br/>
 * <br/>
 *  numexpr   ::=   ( numexpr )<br/>
 *                | NUMBER<br/>
 *                | -numexpr<br/>
 *                | constexpr<br/>
 *                | opexpr<br/>
 *                | varexpr<br/>
 *                | funcexpr<br/>
 *                ;<br/>
 * <br/>
 *  strexpr   ::=   substr ( strexpr , start )<br/>
 *                | substr ( strexpr , start , end )<br/>
 *                | lowercase ( strexpr )<br/>
 *                | uppercase ( strexpr )<br/>
 *                | string<br/>
 *                ;<br/>
 * <br/>
 *  constexpr ::=   PI<br/>
 *                | E<br/>
 *                ;<br/>
 * <br/>
 *  opexpr    ::=   numexpr + numexpr<br/>
 *                | numexpr - numexpr<br/>
 *                | numexpr * numexpr<br/>
 *                | numexpr &#47; numexpr<br/>
 *                | numexpr ^ numexpr<br/>
 *                ;<br/>
 * <br/>
 *  varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 *  funcexpr ::=    abs ( numexpr )<br/>
 *                | sqrt ( numexpr )<br/>
 *                | log ( numexpr )<br/>
 *                | exp ( numexpr )<br/>
 *                | sin ( numexpr )<br/>
 *                | cos ( numexpr )<br/>
 *                | tan ( numexpr )<br/>
 *                | rint ( numexpr )<br/>
 *                | floor ( numexpr )<br/>
 *                | pow ( numexpr , numexpr )<br/>
 *                | ceil ( numexpr )<br/>
 *                | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *                | length ( str )<br/>
 *                ;<br/>
 * <br/>
 * Notes:<br/>
 * - 'start' and 'end' for function 'substr' are indices that start at 1.<br/>
 * - index 'end' for function 'substr' is excluded (like Java's 'String.substring(int,int)' method)<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Sinkhole
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-condition &lt;adams.flow.condition.bool.AbstractBooleanCondition&gt; (property: condition)
 * &nbsp;&nbsp;&nbsp;The condition that determines whether to drop the token (evaluates to 'true'
 * &nbsp;&nbsp;&nbsp;) or not (evaluates to 'false').
 * &nbsp;&nbsp;&nbsp;default: adams.flow.condition.bool.Expression -expression false
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4563 $
 */
public class Sinkhole
  extends AbstractTransformer
  implements ControlActor, GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = -2318544907798411076L;

  /** the key for storing the current input token in the backup. */
  public final static String BACKUP_INPUT = "input";

  /** the condition to evaluate. */
  protected AbstractBooleanCondition m_Condition;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Blocks for propagation of tokens if the condition evaluates to 'true'."
      + "In case of integer or double tokens that arrive at the input, these "
      + "can be accessed in the expression via 'X'.\n\n"
      + "The following grammar is used for evaluating the boolean expressions "
      + "(depends on the selected condition):\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    if (m_Condition instanceof GrammarSupplier)
      return ((GrammarSupplier) m_Condition).getGrammar();
    else
      return null;
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();
    
    m_OptionManager.add(
	    "condition", "condition",
	    getDefaultCondition());
  }
  
  /**
   * Returns the default condition to use.
   * 
   * @return		the condition
   */
  protected AbstractBooleanCondition getDefaultCondition() {
    Expression	result;
    
    result = new Expression();
    result.setExpression(new BooleanExpressionText("false"));
    
    return result;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    return m_Condition.getQuickInfo();
  }

  /**
   * Sets the condition.
   *
   * @param value	the condition
   */
  public void setCondition(AbstractBooleanCondition value) {
    m_Condition = value;
    reset();
  }

  /**
   * Returns the current condition.
   *
   * @return		the condition
   */
  public AbstractBooleanCondition getCondition() {
    return m_Condition;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String conditionTipText() {
    return
        "The condition that determines whether to drop the token "
	+ "(evaluates to 'true') or not (evaluates to 'false').";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_InputToken != null)
      result.put(BACKUP_INPUT, m_InputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_INPUT)) {
      m_InputToken = (Token) state.get(BACKUP_INPUT);
      state.remove(BACKUP_INPUT);
    }

    super.restoreState(state);
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      result = m_Condition.setUp();

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;

    result = null;

    if (!m_Condition.evaluate(this, m_InputToken))
      m_OutputToken = m_InputToken;
    else
      m_OutputToken = null;

    return result;
  }
}
