/*
 * AbstractGraphicalDisplay.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.io.FileUtils;
import adams.core.io.PlaceholderFile;
import adams.flow.core.Token;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.BasePanel;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.print.JComponentWriter;
import adams.gui.print.JComponentWriterFileChooser;
import adams.gui.print.NullWriter;
import adams.gui.print.PNGWriter;
import adams.gui.sendto.SendToActionSupporter;
import adams.gui.sendto.SendToActionUtils;

/**
 * Ancestor for actors that display stuff.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4123 $
 */
public abstract class AbstractGraphicalDisplay
  extends AbstractDisplay
  implements MenuBarProvider, ComponentSupplier, SendToActionSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -4882859296402745911L;

  /** the writer to use. */
  protected JComponentWriter m_Writer;

  /** the menu bar, if used. */
  protected JMenuBar m_MenuBar;

  /** the "clear" menu item. */
  protected JMenuItem m_MenuItemFileClear;

  /** the "save as" menu item. */
  protected JMenuItem m_MenuItemFileSaveAs;

  /** the "save text as" menu item. */
  protected JMenuItem m_MenuItemFileSaveTextAs;

  /** the "exit" menu item. */
  protected JMenuItem m_MenuItemFileClose;

  /** the filedialog for saving the panel as picture. */
  protected JComponentWriterFileChooser m_GraphicFileChooser;

  /** the filedialog for saving the textual data. */
  protected BaseFileChooser m_TextFileChooser;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "writer", "writer",
	    new NullWriter());
  }

  /**
   * Sets the writer.
   *
   * @param value 	the writer
   */
  public void setWriter(JComponentWriter value) {
    m_Writer = value;
    reset();
  }

  /**
   * Returns the writer.
   *
   * @return 		the writer
   */
  public JComponentWriter getWriter() {
    return m_Writer;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String writerTipText() {
    return "The writer to use for generating the graphics output.";
  }

  /**
   * Checks whether a panel is available that can be saved.
   *
   * @return 		true if panel is available for saving
   */
  protected boolean hasPanel() {
    return (supplyComponent() != null);
  }

  /**
   * Returns the current component.
   *
   * @return		the current component, can be null
   */
  public JComponent supplyComponent() {
    return m_Panel;
  }

  /**
   * Hook method before the frame gets created.
   *
   * @param panel	the panel to display in the frame
   */
  protected void preCreateFrame(BasePanel panel) {
    super.preCreateFrame(panel);

    m_GraphicFileChooser = new JComponentWriterFileChooser();
    m_TextFileChooser    = new BaseFileChooser();
    m_TextFileChooser.setAutoAppendExtension(true);
    m_TextFileChooser.addChoosableFileFilter(new ExtensionFileFilter("Text file", "txt"));
  }

  /**
   * Before the token is displayed.
   *
   * @param token	the token to display
   */
  protected void preDisplay(Token token) {
    m_Writer.setComponent(m_Panel);
  }

  /**
   * After the token has been displayed.
   *
   * @param token	the token to display
   */
  protected void postDisplay(Token token) {
    try {
      if (!(m_Writer instanceof NullWriter))
	m_Writer.toOutput();
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
    }
  }

  /**
   * Assembles the menu bar.
   *
   * @return		the menu bar
   */
  protected JMenuBar createMenuBar() {
    JMenuBar	result;
    JMenu	menu;
    JMenuItem	menuitem;

    result = new JMenuBar();

    // File
    menu = new JMenu("File");
    result.add(menu);
    menu.setMnemonic('F');
    menu.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	updateMenu();
      }
    });

    // File/Clear
    if (supportsClear()) {
      menuitem = new JMenuItem("Clear");
      menu.add(menuitem);
      menuitem.setMnemonic('l');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed N"));
      menuitem.setIcon(GUIHelper.getIcon("new.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  clear();
	}
      });
      m_MenuItemFileClear = menuitem;
    }

    // File/Save As
    menuitem = new JMenuItem("Save as...");
    menu.add(menuitem);
    menuitem.setMnemonic('a');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed S"));
    menuitem.setIcon(GUIHelper.getIcon("save.gif"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	saveAs();
      }
    });
    m_MenuItemFileSaveAs = menuitem;

    // File/Save Text As
    if (this instanceof TextSupplier) {
      menuitem = new JMenuItem("Save text as...");
      menu.add(menuitem);
      menuitem.setMnemonic('t');
      menuitem.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          saveTextAs();
        }
      });
      m_MenuItemFileSaveTextAs = menuitem;
    }

    // File/Send to
    menu.addSeparator();
    if (SendToActionUtils.addSendToSubmenu(this, menu))
      menu.addSeparator();

    // File/Close
    menuitem = new JMenuItem("Close");
    menu.add(menuitem);
    menuitem.setMnemonic('C');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Q"));
    menuitem.setIcon(GUIHelper.getIcon("exit.png"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	close();
      }
    });
    m_MenuItemFileClose = menuitem;

    return result;
  }

  /**
   * Creates a menu bar (singleton per panel object). Can be used in frames.
   *
   * @return		the menu bar
   */
  public JMenuBar getMenuBar() {
    if (m_MenuBar == null) {
      m_MenuBar = createMenuBar();
      updateMenu();
    }

    return m_MenuBar;
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateMenu() {
    if (m_MenuBar == null)
      return;

    m_MenuItemFileSaveAs.setEnabled(hasPanel());
    if (m_MenuItemFileSaveTextAs != null)
      m_MenuItemFileSaveTextAs.setEnabled(hasPanel());
  }

  /**
   * Whether "clear" is supported and shows up in the menu.
   * <p/>
   * Default implementation returns "false".
   *
   * @return		true if supported
   */
  protected boolean supportsClear() {
    return false;
  }

  /**
   * Clears the display.
   * <p/>
   * Default implementation does nothing.
   */
  protected void clear() {
  }

  /**
   * Saves the panel as picture.
   */
  protected void saveAs() {
    int			retVal;
    JComponentWriter	writer;

    retVal = m_GraphicFileChooser.showSaveDialog(m_Panel);
    if (retVal != BaseFileChooser.APPROVE_OPTION)
      return;

    writer = m_GraphicFileChooser.getWriter();
    writer.setComponent(supplyComponent());
    try {
      writer.toOutput();
    }
    catch (Exception e) {
      getSystemErr().println("Error saving panel to '" + writer.getFile() + "': ");
      getSystemErr().printStackTrace(e);
    }
  }

  /**
   * Saves the text.
   */
  protected void saveTextAs() {
    int		retVal;
    boolean	ok;

    retVal = m_TextFileChooser.showSaveDialog(m_Panel);
    if (retVal != BaseFileChooser.APPROVE_OPTION)
      return;

    ok = FileUtils.writeToFile(
	m_TextFileChooser.getSelectedFile().getAbsolutePath(),
	((TextSupplier) this).supplyText(),
	false);

    if (!ok)
      getSystemErr().println("Error saving ext to '" + m_TextFileChooser.getSelectedFile() + "'!");
  }

  /**
   * Closes the dialog or frame.
   */
  protected void close() {
    m_Panel.closeParent();
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    super.cleanUpGUI();

    m_MenuBar                = null;
    m_MenuItemFileClear      = null;
    m_MenuItemFileSaveAs     = null;
    m_MenuItemFileSaveTextAs = null;
    m_MenuItemFileClose      = null;
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  public Class[] getSendToClasses() {
    return new Class[]{PlaceholderFile.class, JComponent.class};
  }

  /**
   * Checks whether something to send is available.
   *
   * @param cls		the requested classes
   * @return		true if an object is available for sending
   */
  public boolean hasSendToItem(Class[] cls) {
    return    (SendToActionUtils.isAvailable(new Class[]{PlaceholderFile.class, JComponent.class}, cls))
           && (supplyComponent() != null);
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the requested classes
   * @return		the item to send
   */
  public Object getSendToItem(Class[] cls) {
    Object		result;
    JComponent		comp;
    PNGWriter		writer;

    result = null;

    comp = supplyComponent();
    if (SendToActionUtils.isAvailable(PlaceholderFile.class, cls)) {
      if (comp != null) {
	result = SendToActionUtils.nextTmpFile("actor-" + getName(), "png");
	writer = new PNGWriter();
	writer.setFile((PlaceholderFile) result);
	writer.setComponent(comp);
	try {
	  writer.generateOutput();
	}
	catch (Exception e) {
	  getSystemErr().println("Failed to write image to " + result + ":");
	  getSystemErr().printStackTrace(e);
	  result = null;
	}
      }
    }
    else if (SendToActionUtils.isAvailable(JComponent.class, cls)) {
      result = comp;
    }

    return result;
  }
}
