/*
 * SequenceToArray.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.lang.reflect.Array;
import java.util.Hashtable;
import java.util.Vector;

import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Turns a sequence of tokens into arrays with a specified length.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown[]<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SequenceToArray
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-length &lt;int&gt; (property: arrayLength)
 * &nbsp;&nbsp;&nbsp;The length of the arrays.
 * &nbsp;&nbsp;&nbsp;default: 1
 * </pre>
 *
 * <pre>-array-class &lt;java.lang.String&gt; (property: arrayClass)
 * &nbsp;&nbsp;&nbsp;The class to use for the array; if none is specified, the class of the first
 * &nbsp;&nbsp;&nbsp;element is used.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2414 $
 */
public class SequenceToArray
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 8411367398473311627L;

  /** the key for storing the current elements in the backup. */
  public final static String BACKUP_ELEMENTS = "elements";

  /** the buffered elements of the array that still need to be broadcasted. */
  protected Vector m_Elements;

  /** the length of the arrays. */
  protected int m_ArrayLength;

  /** the class for the array. */
  protected String m_ArrayClass;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Turns a sequence of tokens into arrays with a specified length.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "length", "arrayLength",
	    1);

    m_OptionManager.add(
	    "array-class", "arrayClass",
	    "");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("arrayLength");
    if (variable != null)
      result = "Length: " + variable;
    else
      result = "Length: " + m_ArrayLength;

    result += ", Class: ";
    variable = getOptionManager().getVariableForProperty("arrayClass");
    if (variable != null)
      result += variable;
    else if (m_ArrayClass.length() != 0)
      result += m_ArrayClass;
    else
      result += "-from 1st element-";

    return result;
  }

  /**
   * Sets the length of the arrays.
   *
   * @param value	the length
   */
  public void setArrayLength(int value) {
    if (value < 1) {
      getSystemErr().println("Arrays must have a length of at least 1, provided: " + value + "!");
    }
    else {
      m_ArrayLength = value;
      reset();
    }
  }

  /**
   * Returns the length of the arrays.
   *
   * @return		the length
   */
  public int getArrayLength() {
    return m_ArrayLength;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String arrayLengthTipText() {
    return "The length of the arrays.";
  }

  /**
   * Sets the class for the array.
   *
   * @param value	the classname, use empty string to use class of first
   * 			element
   */
  public void setArrayClass(String value) {
    m_ArrayClass = value;
    reset();
  }

  /**
   * Returns the class for the array.
   *
   * @return		the classname, empty string if class of first element
   * 			is used
   */
  public String getArrayClass() {
    return m_ArrayClass;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String arrayClassTipText() {
    return
        "The class to use for the array; if none is specified, the class of "
      + "the first element is used.";
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_ELEMENTS);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_ELEMENTS, m_Elements);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_ELEMENTS)) {
      m_Elements = (Vector) state.get(BACKUP_ELEMENTS);
      state.remove(BACKUP_ELEMENTS);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Elements    = new Vector();
    m_OutputToken = null;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown[].class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown[].class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    int		i;
    Object	array;

    result = null;

    try {
      m_Elements.add(m_InputToken.getPayload());
      debug("Buffered elements: " + m_Elements.size());
      if (m_Elements.size() == m_ArrayLength) {
	if (m_ArrayClass.length() == 0)
	  array = Array.newInstance(m_Elements.firstElement().getClass(), m_ArrayLength);
	else
	  array = Array.newInstance(Class.forName(m_ArrayClass), m_ArrayLength);
	debug("Array type: " + array.getClass().getComponentType());
	for (i = 0; i < m_ArrayLength; i++)
	  Array.set(array, i, m_Elements.get(i));
	m_OutputToken = new Token(array);
	m_Elements.clear();
	debug("Array generated");
      }
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
