/**
 * AbstractInitialization.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.application;

import adams.core.ClassLister;
import adams.core.option.OptionUtils;

/**
 * Ancestor for initialization applets.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4472 $
 */
public abstract class AbstractInitialization {

  /**
   * The title of the initialization.
   * 
   * @return		the title
   */
  public abstract String getTitle();
  
  /**
   * Performs the initialization.
   * 
   * @param parent	the application this initialization is for
   * @return		true if successful
   */
  public abstract boolean initialize(AbstractApplicationFrame parent);

  /**
   * Runs all the initialization applets.
   * 
   * @param parent	the application this initialization is for
   * @return		true if all successful
   */
  public static boolean initAll(AbstractApplicationFrame parent) {
    boolean			result;
    String[]			classes;
    AbstractInitialization	init;
    
    result = true;
    classes = AbstractInitialization.getInitializations();
    for (String cls: classes) {
      try {
	init   = (AbstractInitialization) Class.forName(cls).newInstance();
	result = init.initialize(parent) || result;
      }
      catch (Exception e) {
	System.err.println("Failed to run initialization applet '" + cls + "':");
	e.printStackTrace();
      }
    }
    
    return result;
  }
  
  /**
   * Returns a list with classnames of initialization applets.
   *
   * @return		the initialization applet classnames
   */
  public static String[] getInitializations() {
    return ClassLister.getSingleton().getClassnames(AbstractInitialization.class);
  }

  /**
   * Instantiates the initialization applet with the (optional) options.
   *
   * @param classname	the classname of the initialization applet to instantiate
   * @param options	the options for the initialization applet
   * @return		the instantiated initialization applet or null if an error occurred
   */
  public static AbstractInitialization forName(String classname, String[] options) {
    AbstractInitialization	result;

    try {
      result = (AbstractInitialization) OptionUtils.forName(AbstractInitialization.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }
}
