/*
 * NotesFactory.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.container;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Dialog.ModalityType;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTextPane;
import javax.swing.text.Document;

import adams.core.Properties;
import adams.data.Notes;
import adams.data.NotesHandler;
import adams.data.id.DatabaseIDHandler;
import adams.data.report.ReportHandler;
import adams.env.Environment;
import adams.env.ScriptingDialogDefinition;
import adams.gui.core.BaseDialog;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.BaseTabbedPane;
import adams.gui.core.GUIHelper;
import adams.gui.scripting.SyntaxDocument;

/**
 * A factory for GUI components for notes.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4095 $
 */
public class NotesFactory {

  /**
   * A specialized JTextPane for displaying the notes of a spectrum.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4095 $
   * @param <T> the type of container to use
   */
  public static class TextPane<T extends Container>
    extends JTextPane {

    /** for serialization. */
    private static final long serialVersionUID = -1236250353157866097L;

    /** the underlying data. */
    protected T m_Data;

    /**
     * Initializes the text pane with no notes.
     */
    public TextPane() {
      this(null);
    }

    /**
     * Initializes the text pane with the notes in the given spectrum.
     *
     * @param data	the data to initialize with
     */
    public TextPane(T data) {
      super();

      setDocument(createDocument());
      setEditable(false);

      setData(data);
    }

    /**
     * Creates a new document for the dialog, with syntax highlighting support.
     *
     * @return		the new document
     */
    protected Document createDocument() {
      Document	result;
      Properties	props;

      props  = Environment.getInstance().read(ScriptingDialogDefinition.KEY);
      result = new SyntaxDocument(props);

      return result;
    }

    /**
     * Sets the data and spectrum notes.
     *
     * @param data	the spectrum container containing the spectrum notes
     */
    public void setData(T data) {
      Notes		notes;
      StringBuffer	buffer;
      Notes		subnotes;

      m_Data = data;

      if (m_Data.getPayload() instanceof NotesHandler) {
	notes  = ((NotesHandler) m_Data.getPayload()).getNotes();
	buffer = new StringBuffer();

	// errors
	subnotes = notes.getErrors();
	buffer.append("Errors\n");
	if (subnotes.size() == 0)
	  buffer.append("-none-");
	else
	  buffer.append(subnotes.toString());
	buffer.append("\n");

	// warnings
	subnotes = notes.getWarnings();
	buffer.append("\n");
	buffer.append("Warnings\n");
	if (subnotes.size() == 0)
	  buffer.append("-none-");
	else
	  buffer.append(subnotes.toString());
	buffer.append("\n");

	// others
	subnotes = notes.getOthers();
	buffer.append("\n");
	buffer.append("Others\n");
	if (subnotes.size() == 0)
	  buffer.append("-none-");
	else
	  buffer.append(subnotes.toString());
	buffer.append("\n");

	// process information
	subnotes = notes.getProcessInformation();
	buffer.append("\n");
	buffer.append("Process information\n");
	if (subnotes.size() == 0)
	  buffer.append("-none-");
	else
	  buffer.append(subnotes.toString().replaceAll(Notes.PROCESS_INFORMATION + "[^\n]*\n", ""));
	buffer.append("\n");

	setText("");
	try {
	  getDocument().insertString(0, buffer.toString(), null);
	}
	catch (Exception e) {
	  e.printStackTrace();
	}
      }
      else {
	setText("");
      }
    }

    /**
     * Returns the underlying data.
     *
     * @return		the spectrum container
     */
    public T getData() {
      return m_Data;
    }

    /**
     * Sets the size of the text pane. Gets adapted.
     *
     * @param d		the dimension
     */
    public void setSize(Dimension d) {
      if (d.width < getGraphicsConfiguration().getBounds().width)
        d.width = getGraphicsConfiguration().getBounds().width;
      super.setSize(d);
    }

    /**
     * Always returns false.
     *
     * @return		always false
     */
    public boolean getScrollableTracksViewportWidth() {
      return false;
    }
  }

  /**
   * A specialized tabbed pane that displays spectrum notes.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4095 $
   * @see QuantitationReport
   * @param <T> the type of container to use
   */
  public static class TabbedPane<T extends Container>
    extends BaseTabbedPane {

    /** for serialization. */
    private static final long serialVersionUID = -113778971321461204L;

    /** the underlying data. */
    protected Vector<T> m_Data;

    /**
     * Initializes the tabbed pane with not notes.
     */
    public TabbedPane() {
      this(null);
    }

    /**
     * Initializes the tabbed pane with the given spectrum notes.
     *
     * @param data	the spectrum containers containing the spectrum notes
     */
    public TabbedPane(Vector<T> data) {
      super();

      setData(data);
    }

    /**
     * Sets the data and spectrum notes.
     *
     * @param data	the spectrum containers containing the spectrum notes
     */
    public synchronized void setData(Vector<T> data) {
      int	i;

      m_Data = new Vector<T>();
      if (data != null) {
	for (i = 0; i < data.size(); i++)
	  m_Data.add(data.get(i));
      }
      update();
    }

    /**
     * Returns the underlying data.
     *
     * @return		the spectrum containers
     */
    public Vector<T> getData() {
      Vector<T>	result;

      result = new Vector<T>();
      result.addAll(m_Data);

      return result;
    }

    /**
     * updates the tabbed pane.
     */
    protected synchronized void update() {
      int		i;
      TextPane		text;
      ReportHandler	handler;
      String		title;

      removeAll();

      for (i = 0; i < m_Data.size(); i++) {
	if (!(m_Data.get(i).getPayload() instanceof ReportHandler))
	  continue;
	handler = (ReportHandler) m_Data.get(i).getPayload();
        if (!handler.hasReport())
          continue;

        text = getTextPane(m_Data.get(i));
        if (m_Data.get(i) instanceof NamedContainer) {
          title = ((NamedContainer) m_Data.get(i)).getID();
          if (m_Data.get(i).getPayload() instanceof DatabaseIDHandler)
            title += " (" +  ((DatabaseIDHandler) m_Data.get(i).getPayload()).getDatabaseID() + ")";
        }
        else {
          title = m_Data.get(i).toString();
        }
        addTab(title, new BaseScrollPane(text));
      }
    }
  }

  /**
   * A specialized dialog that displays informative statistics.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4095 $
   * @param <T> the type of container to use
   */
  public static class Dialog<T extends Container>
    extends BaseDialog {

    /** for serialization. */
    private static final long serialVersionUID = 1739123906988587074L;

    /** the default width. */
    public final static int DEFAULT_WIDTH = 600;

    /** the minimum height. */
    public final static int MIN_HEIGHT = 200;

    /** the dialog itself. */
    protected Dialog m_Self;

    /** the tabbed pane for displaying the statistics. */
    protected TabbedPane m_TabbedPane;

    /**
     * Initializes the dialog.
     *
     * @param owner	the component that controls the dialog
   * @param modality	the type of modality
     */
    public Dialog(java.awt.Dialog owner, ModalityType modality) {
      super(owner, modality);
    }

    /**
     * Initializes the dialog.
     *
     * @param owner	the component that controls the dialog
     * @param modal	if true then the dialog will be modal
     */
    public Dialog(java.awt.Frame owner, boolean modal) {
      super(owner, modal);
    }

    /**
     * For initializing members.
     */
    protected void initialize() {
      super.initialize();

      m_Self = this;
    }

    /**
     * Initializes the components.
     */
    protected void initGUI() {
      JPanel	panel;
      JButton	buttonOK;

      super.initGUI();

      setTitle("Notes");
      getContentPane().setLayout(new BorderLayout());

      // tabbed pane
      m_TabbedPane = getTabbedPane(null);
      getContentPane().add(m_TabbedPane, BorderLayout.CENTER);

      // OK button
      panel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
      getContentPane().add(panel, BorderLayout.SOUTH);

      buttonOK = new JButton("OK");
      buttonOK.setMnemonic('O');
      buttonOK.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          m_Self.setVisible(false);
        }
      });
      panel.add(buttonOK);

      adjustSize();
      setLocationRelativeTo(getOwner());
    }

    /**
     * Adjusts the size of the dialog.
     */
    protected void adjustSize() {
      int	height;

      pack();

      height = (int) getSize().getHeight();
      if (height < MIN_HEIGHT)
	height = MIN_HEIGHT;

      setSize(new Dimension(DEFAULT_WIDTH, height));
    }

    /**
     * Sets the data to display.
     *
     * @param value	the underlying containers
     */
    public synchronized void setData(Vector<T> value) {
      m_TabbedPane.setData(value);

      if (!isVisible()) {
	adjustSize();
	setLocationRelativeTo(getOwner());
	GUIHelper.setSizeAndLocation(this, this);
      }
    }

    /**
     * Returns the underlying data.
     *
     * @return		the containers
     */
    public Vector<T> getData() {
      return m_TabbedPane.getData();
    }
  }

  /**
   * Returns a new table for the given spectrum notes.
   *
   * @param data	the spectrums to create a tabbed pane for
   * @return		the tabbed pane
   */
  public static TextPane getTextPane(Container data) {
    return new TextPane(data);
  }

  /**
   * Returns a new table for the given spectrum notes.
   *
   * @param data	the spectrums to create a tabbed pane for
   * @return		the tabbed pane
   */
  public static TabbedPane getTabbedPane(Vector<Container> data) {
    return new TabbedPane(data);
  }

  /**
   * Returns a new dialog for displaying spectrum notes.
   *
   * @param owner	the owning component
   * @param modality	the type of modality
   * @return		the dialog
   */
  public static Dialog getDialog(java.awt.Dialog owner, ModalityType modality) {
    return new Dialog(owner, modality);
  }

  /**
   * Returns a new dialog for displaying spectrum notes.
   *
   * @param owner	the owning component
   * @param modal	if true then the dialog will be modal
   * @return		the dialog
   */
  public static Dialog getDialog(java.awt.Frame owner, boolean modal) {
    return new Dialog(owner, modal);
  }
}
