/*
 * Paintlet.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core;


import java.awt.BasicStroke;
import java.awt.Graphics;
import java.awt.Graphics2D;

import adams.core.ClassLister;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.AbstractOptionHandler;
import adams.core.option.OptionUtils;
import adams.gui.event.PaintEvent.PaintMoment;

/**
 * An abstract superclass for paint engines that can be plugged into panels.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 */
public abstract class AbstractPaintlet
  extends AbstractOptionHandler
  implements Paintlet {

  /** for serialization. */
  private static final long serialVersionUID = -8877675053270937462L;

  /** the panel this paintlet is for. */
  protected PaintablePanel m_Panel;

  /** whether the paintlet is enabled. */
  protected boolean m_Enabled;

  /** whether the paintlet reacts with repaints to changes of its members. */
  protected boolean m_RepaintOnChange;

  /** the thickness of the stroke. */
  protected float m_StrokeThickness;

  /** whether the paintlet is currently being initialized and should ignore
   * repaint requests. */
  protected boolean m_Initializing;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "stroke-thickness", "strokeThickness",
	    1.0f, 0.01f, null);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Panel           = null;
    m_Initializing    = true;
    m_Enabled         = true;
    m_RepaintOnChange = true;
  }

  /**
   * Finishes the initialization in the constructor.
   * <p/>
   * Calls memberChanged.
   *
   * @see		#memberChanged()
   */
  protected void finishInit() {
    super.finishInit();

    m_Initializing = false;
    memberChanged();
  }

  /**
   * Whether the paintlet is currently being initialized.
   *
   * @return		true if the paintlet is currently being initialized
   */
  protected boolean isInitializing() {
    return m_Initializing;
  }

  /**
   * Sets the panel to use, null to disable painting.
   *
   * @param value	the panel to paint on
   */
  public void setPanel(PaintablePanel value) {
    if (value != m_Panel) {
      // remove old registration
      if (m_Panel != null)
	m_Panel.removePaintlet(this);

      m_Panel = value;

      // register with new panel
      if (m_Panel != null)
	m_Panel.addPaintlet(this);

      memberChanged(true);
    }
  }

  /**
   * Returns the panel currently in use.
   *
   * @return		the panel in use
   */
  public PaintablePanel getPanel() {
    return m_Panel;
  }

  /**
   * Returns whether a panel has been set.
   *
   * @return		true if a panel is currently set
   */
  public boolean hasPanel() {
    return (m_Panel != null);
  }

  /**
   * Returns the plot panel of the panel, null if no panel present.
   *
   * @return		the plot panel
   */
  public PlotPanel getPlot() {
    PlotPanel	result;

    result = null;

    if (m_Panel != null)
      result = m_Panel.getPlot();

    return result;
  }

  /**
   * Sets whether the paintlet is enabled or not. Setting it to true
   * automatically initiates a repaint. Is not affected by m_RepaintOnChange.
   *
   * @param value	if true then the paintlet is enabled
   * @see		#m_RepaintOnChange
   */
  public void setEnabled(boolean value) {
    m_Enabled = value;
    repaint();
  }

  /**
   * Returns whether the paintlet is currently enabled.
   *
   * @return		true if the paintlet is enabled.
   */
  public boolean isEnabled() {
    return m_Enabled;
  }

  /**
   * Sets whether the paintlet reacts with repaints to changes of its members.
   *
   * @param value	if true then the paintlet repaints whenever members
   * 			get changed
   */
  public void setRepaintOnChange(boolean value) {
    m_RepaintOnChange = value;
  }

  /**
   * Returns whether the paintlet reacts with repaints to changes of its members.
   *
   * @return		true if paintlet repaints whenever members get changed
   */
  public boolean getRepaintOnChange() {
    return m_RepaintOnChange;
  }

  /**
   * Sets the stroke thickness to use.
   *
   * @param value	the thickness
   */
  public void setStrokeThickness(float value) {
    m_StrokeThickness = value;
    memberChanged();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String strokeThicknessTipText() {
    return "The thickness of the stroke.";
  }

  /**
   * Returns the current stroke thickness.
   *
   * @return		the thickness
   */
  public float getStrokeThickness() {
    return m_StrokeThickness;
  }

  /**
   * Returns when this paintlet is to be executed.
   *
   * @return		when this paintlet is to be executed
   */
  public abstract PaintMoment getPaintMoment();

  /**
   * The paint routine of the paintlet.
   *
   * @param g		the graphics context to use for painting
   */
  public abstract void performPaint(Graphics g);

  /**
   * The paint routine of the paintlet.
   *
   * @param g		the graphics context to use for painting
   * @see		#isEnabled()
   */
  public void paint(Graphics g) {
    Graphics2D 	g2d;
    float	width;

    if (isEnabled() && hasPanel()) {
      g2d   = null;
      width = 1.0f;

      // set width
      if (g instanceof Graphics2D) {
	g2d = (Graphics2D) g;
	if (g2d.getStroke() instanceof BasicStroke)
	  width = ((BasicStroke) g2d.getStroke()).getLineWidth();
	g2d.setStroke(new BasicStroke(m_StrokeThickness));
      }

      // paint
      performPaint(g);

      // restore width
      if (g2d != null)
	g2d.setStroke(new BasicStroke(width));
    }
  }

  /**
   * Calls the update() method of the associated panel.
   */
  protected void updatePanel() {
    if (hasPanel())
      getPanel().update();
  }

  /**
   * Executes a repaints only if the changes to members are not ignored.
   *
   * @see		#getRepaintOnChange()
   * @see		#isInitializing()
   * @see		#repaint()
   */
  public void memberChanged() {
    memberChanged(false);
  }

  /**
   * Executes a repaints only if the changes to members are not ignored.
   *
   * @param updatePanel	whether to call the update() method of the associated
   * 			panel
   * @see		#getRepaintOnChange()
   * @see		#isInitializing()
   * @see		#repaint()
   */
  public void memberChanged(boolean updatePanel) {
    reset();

    if (updatePanel)
      updatePanel();

    if (!isInitializing()) {
      if (getRepaintOnChange())
	repaint();
    }
  }

  /**
   * Repaints if possible (and enabled!).
   *
   * @see		#m_Panel
   * @see		#isEnabled()
   */
  public void repaint() {
    if (isEnabled() && hasPanel())
      m_Panel.update();
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @return		the shallow copy
   */
  public AbstractPaintlet shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractPaintlet shallowCopy(boolean expand) {
    return (AbstractPaintlet) OptionUtils.shallowCopy(this, expand);
  }

  /**
   * Returns a list with classnames of paintlets.
   *
   * @return		the filter classnames
   */
  public static String[] getPaintlets() {
    return ClassLister.getSingleton().getClassnames(AbstractPaintlet.class);
  }

  /**
   * Instantiates the paintlet with the given options.
   *
   * @param classname	the classname of the paintlet to instantiate
   * @param options	the options for the paintlet
   * @return		the instantiated paintlet or null if an error occurred
   */
  public static AbstractPaintlet forName(String classname, String[] options) {
    AbstractPaintlet	result;

    try {
      result = (AbstractPaintlet) OptionUtils.forName(AbstractPaintlet.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the paintlet from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			paintlet to instantiate
   * @return		the instantiated paintlet
   * 			or null if an error occurred
   */
  public static AbstractPaintlet forCommandLine(String cmdline) {
    return (AbstractPaintlet) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }
}
