/*
 * PaintablePanel.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core;


import java.awt.Graphics;
import java.util.HashSet;
import java.util.Iterator;

import javax.swing.SwingUtilities;

import adams.gui.core.BasePanel;
import adams.gui.event.PaintEvent;
import adams.gui.event.PaintListener;
import adams.gui.event.PaintEvent.PaintMoment;

/**
 * An abstract superclass for panels that paint "stuff", e.g., spectrums.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3468 $
 */
public abstract class PaintablePanel
  extends BasePanel
  implements PaintListener {

  /** for serialization. */
  private static final long serialVersionUID = -1394066820727332049L;

  /** additional paintlets to execute. */
  protected HashSet<AbstractPaintlet> m_Paintlets;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Paintlets = new HashSet<AbstractPaintlet>();
  }

  /**
   * Adds the paintlet to the internal list.
   *
   * @param p		the paintlet to add.
   */
  public void addPaintlet(AbstractPaintlet p) {
    synchronized(m_Paintlets) {
      m_Paintlets.add(p);
    }
  }

  /**
   * Removes this paintlet from its internal list.
   *
   * @param p		the paintlet to remove
   */
  public void removePaintlet(AbstractPaintlet p) {
    synchronized(m_Paintlets) {
      m_Paintlets.remove(p);
    }
  }

  /**
   * Returns an iterator over all currently stored paintlets.
   *
   * @return		the paintlets
   */
  public Iterator<AbstractPaintlet> paintlets() {
    synchronized(m_Paintlets) {
      return m_Paintlets.iterator();
    }
  }

  /**
   * Returns the plot panel of the panel, null if no panel present.
   *
   * @return		the plot panel
   */
  public abstract PlotPanel getPlot();

  /**
   * Prepares the update, i.e., calculations etc.
   */
  protected abstract void prepareUpdate();

  /**
   * Performs the actual update.
   */
  protected void performUpdate() {
    Runnable	run;

    run = new Runnable() {
      public void run() {
	validate();
	repaint();
      }
    };
    SwingUtilities.invokeLater(run);
  }

  /**
   * Updates the display.
   */
  public synchronized void update() {
    prepareUpdate();
    performUpdate();
  }

  /**
   * Returns true if the paintlets can be executed.
   *
   * @param g		the graphics context
   * @return		true if painting can go ahead
   */
  protected abstract boolean canPaint(Graphics g);

  /**
   * Executes all paintlets of the specified paint moment.
   *
   * @param g		the graphics context
   * @param moment	the paint moment
   */
  protected void paint(Graphics g, PaintMoment moment) {
    Iterator<AbstractPaintlet>	iter;
    AbstractPaintlet		paintlet;

    synchronized(m_Paintlets) {
      iter = m_Paintlets.iterator();
      while (iter.hasNext()) {
	paintlet = iter.next();
	if (paintlet.getPaintMoment() == moment)
	  paintlet.paint(g);
      }
    }
  }

  /**
   * draws the data.
   *
   * @param e		the paint event
   */
  public void painted(PaintEvent e) {
    Graphics	g;

    g = e.getGraphics();

    // not yet ready?
    if (!canPaint(g))
      return;

    // execute paintlets
    paint(g, e.getPaintMoment());
  }
}
