/*
 * HistogramFactory.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.statistics;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Dialog.ModalityType;

import org.math.plot.Plot2DPanel;
import org.math.plot.PlotPanel;
import org.math.plot.plots.HistogramPlot2D;

import adams.data.statistics.AbstractArrayStatistic;
import adams.data.statistics.ArrayHistogram;
import adams.data.statistics.StatUtils;
import adams.data.statistics.AbstractArrayStatistic.StatisticContainer;
import adams.gui.core.BaseDialog;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseTabbedPane;
import adams.gui.goe.GenericObjectEditorDialog;

/**
 * A factory for histogram related objects.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4095 $
 */
public class HistogramFactory {

  /** the default number of bins. */
  public final static int NUM_BINS = 50;

  /** whether to use JMathPlot's BarPlot instead of HistogramPlot. */
  public final static boolean USE_BARPLOT = false;

  /**
   * A panel for displaying a histogram based on the GC data of a chromatogram.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4095 $
   */
  public static class Panel
    extends BasePanel {

    /** for serialization. */
    private static final long serialVersionUID = -1990327944906647098L;

    /** the plot panel. */
    protected Plot2DPanel m_PlotPanel;

    /**
     * For initializing the GUI.
     */
    protected void initGUI() {
      super.initGUI();

      setLayout(new BorderLayout());
      m_PlotPanel = new Plot2DPanel();
      add(m_PlotPanel, BorderLayout.CENTER);
    }

    /**
     * Generates from the specified data using the configured ArrayHistogram
     * object and adds the plot.
     *
     * @param hist	for generating the histogram data
     * @param data	the data to generate the histogram from
     * @param name	the name of the tab
     */
    public void add(ArrayHistogram hist, double[] data, String name) {
      add(hist, StatUtils.toNumberArray(data), name);
    }

    /**
     * Generates from the specified data using the configured ArrayHistogram
     * object and adds the plot.
     *
     * @param hist	for generating the histogram data
     * @param data	the data to generate the histogram from
     * @param name	the name of the tab
     */
    public void add(ArrayHistogram hist, Number[] data, String name) {
      ArrayHistogram		histogram;
      StatisticContainer 	cont;
      double[][] 		plotdata;
      int			numBins;
      double[] 			binX;
      double 			binWidth;
      int			i;
      HistogramPlot2D 		plot;
      Color			color;

      // generate the histogram data
      histogram = (ArrayHistogram) hist.shallowCopy(true);
      histogram.add(data);
      cont = histogram.calculate();

      // generate data for plot
      numBins  = (Integer) cont.getMetaData(ArrayHistogram.METADATA_NUMBINS);
      binX     = (double[]) cont.getMetaData(ArrayHistogram.METADATA_BINX);
      binWidth = (Double) cont.getMetaData(ArrayHistogram.METADATA_BINWIDTH);
      plotdata = new double[numBins][2];
      for (i = 0; i < plotdata.length; i++) {
        plotdata[i][1] = (Double) cont.getCell(0, i);
        plotdata[i][0] = binX[i] + binWidth / 2;
      }

      // create plot
      color = PlotPanel.COLORLIST[m_PlotPanel.getPlots().size() % PlotPanel.COLORLIST.length];
      plot  = new HistogramPlot2D(name, color, plotdata, binWidth);
      m_PlotPanel.addPlot(plot);
      m_PlotPanel.setFixedBounds(0, 0, numBins);

      histogram.destroy();
    }
  }

  /**
   * A dialog that queries the user about parameters for displaying histograms.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4095 $
   */
  public static class SetupDialog
    extends GenericObjectEditorDialog {

    /** for serialization. */
    private static final long serialVersionUID = 4215632076348292959L;

    /**
     * Initializes the dialog.
     *
     * @param owner	the component that controls the dialog
   * @param modality	the type of modality
     */
    public SetupDialog(java.awt.Dialog owner, ModalityType modality) {
      super(owner, modality);
    }

    /**
     * Initializes the dialog.
     *
     * @param owner	the component that controls the dialog
     * @param modal	if true then the dialog will be modal
     */
    public SetupDialog(java.awt.Frame owner, boolean modal) {
      super(owner, modal);
    }

    /**
     * Initializes the members.
     */
    protected void initialize() {
      super.initialize();

      getGOEEditor().setClassType(AbstractArrayStatistic.class);
      getGOEEditor().setCanChangeClassInDialog(false);
      setCurrent(new ArrayHistogram());
    }
  }

  /**
   * Dialog for displaying histograms.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4095 $
   */
  public static class Dialog
    extends BaseDialog {

    /** for serialization. */
    private static final long serialVersionUID = 2464157048335973279L;

    /** the tabbed pane for displaying the histograms. */
    protected BaseTabbedPane m_TabbedPane;

    /**
     * Initializes the dialog.
     *
     * @param owner	the component that controls the dialog
   * @param modality	the type of modality
     */
    public Dialog(java.awt.Dialog owner, ModalityType modality) {
      super(owner, modality);
    }

    /**
     * Initializes the dialog.
     *
     * @param owner	the component that controls the dialog
     * @param modal	if true then the dialog will be modal
     */
    public Dialog(java.awt.Frame owner, boolean modal) {
      super(owner, modal);
    }

    /**
     * initializes the GUI.
     */
    protected void initGUI() {
      super.initGUI();

      setTitle("Histogram");
      getContentPane().setLayout(new BorderLayout());
      m_TabbedPane = new BaseTabbedPane();
      getContentPane().add(m_TabbedPane, BorderLayout.CENTER);

      setSize(new Dimension(800, 600));
    }

    /**
     * Removes all the tabs.
     */
    public void clear() {
      m_TabbedPane.removeAll();
    }

    /**
     * Generates from the specified data using the configured ArrayHistogram
     * object and adds the plot.
     *
     * @param hist	for generating the histogram data
     * @param data	the data to generate the histogram from
     * @param name	the name of the tab
     */
    public void add(ArrayHistogram hist, double[] data, String name) {
      add(hist, StatUtils.toNumberArray(data), name);
    }

    /**
     * Generates from the specified data using the configured ArrayHistogram
     * object and adds the plot.
     *
     * @param hist	for generating the histogram data
     * @param data	the data to generate the histogram from
     * @param name	the name of the tab
     */
    public void add(ArrayHistogram hist, Number[] data, String name) {
      Panel	panel;

      panel = new HistogramFactory.Panel();
      panel.add(hist, data, name);
      m_TabbedPane.addTab(name, panel);
    }
  }

  /**
   * Returns an instance of a new panel for displaying histograms.
   *
   * @return		the panel
   */
  public static Panel getPanel() {
    return new Panel();
  }

  /**
   * Returns an instance of a setup dialog for displaying histograms.
   *
   * @param owner	the owning component
   * @param modality	the type of modality
   * @return		the dialog
   */
  public static SetupDialog getSetupDialog(java.awt.Dialog owner, ModalityType modality) {
    return new SetupDialog(owner, modality);
  }

  /**
   * Returns an instance of a setup dialog for displaying histograms.
   *
   * @param owner	the owning component
   * @param modal	if true then the dialog will be modal
   * @return		the dialog
   */
  public static SetupDialog getSetupDialog(java.awt.Frame owner, boolean modal) {
    return new SetupDialog(owner, modal);
  }

  /**
   * Returns a new dialog for displaying histograms.
   *
   * @param owner	the owning component
   * @param modality	the type of modality
   * @return		the dialog
   */
  public static Dialog getDialog(java.awt.Dialog owner, ModalityType modality) {
    return new Dialog(owner, modality);
  }

  /**
   * Returns a new dialog for displaying displaying histograms.
   *
   * @param owner	the owning component
   * @param modal	if true then the dialog will be modal
   * @return		the dialog
   */
  public static Dialog getDialog(java.awt.Frame owner, boolean modal) {
    return new Dialog(owner, modal);
  }
}
