/*
 * MathematicalExpression.java
 * Copyright (C) 2008-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.parser;

import java.io.ByteArrayInputStream;
import java.util.HashMap;

import java_cup.runtime.DefaultSymbolFactory;
import java_cup.runtime.SymbolFactory;
import adams.parser.mathematicalexpression.Parser;
import adams.parser.mathematicalexpression.Scanner;

/**
 <!-- globalinfo-start -->
 * Evaluates mathematical expressions.<br/>
 * <br/>
 * The following grammar is used:<br/>
 * <br/>
 * expr_list ::= expr_list expr_part | expr_part ;<br/>
 * expr_part ::= expr ;<br/>
 * <br/>
 * numexpr   ::=   ( numexpr )<br/>
 *               |  NUMBER<br/>
 *               | -numexpr<br/>
 *               | constexpr<br/>
 *               | opexpr<br/>
 *               | varexpr<br/>
 *               | funcexpr<br/>
 *               ;<br/>
 * <br/>
 * constexpr ::=   PI<br/>
 *               | E<br/>
 *               ;<br/>
 * <br/>
 * opexpr    ::=   numexpr + numexpr<br/>
 *               | numexpr - numexpr<br/>
 *               | numexpr * numexpr<br/>
 *               | numexpr &#47; numexpr<br/>
 *               | numexpr ^ numexpr<br/>
 *               ;<br/>
 * <br/>
 * varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 * funcexpr ::=    abs ( numexpr )<br/>
 *               | sqrt ( numexpr )<br/>
 *               | log ( numexpr )<br/>
 *               | exp ( numexpr )<br/>
 *               | sin ( numexpr )<br/>
 *               | cos ( numexpr )<br/>
 *               | tan ( numexpr )<br/>
 *               | rint ( numexpr )<br/>
 *               | floor ( numexpr )<br/>
 *               | pow ( numexpr , numexpr )<br/>
 *               | ceil ( numexpr )<br/>
 *               | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *               | ifmissing ( variable , numexpr (default value if variable is missing) )<br/>
 *               ;<br/>
 * <br/>
 * boolexpr ::=    BOOLEAN<br/>
 *               | true<br/>
 *               | false<br/>
 *               | istrue ( VARIABLE (must represent a boolean string: true|false) )<br/>
 *               | numexpr &amp;lt; numexpr<br/>
 *               | numexpr &amp;lt;= numexpr<br/>
 *               | numexpr &amp;gt; numexpr<br/>
 *               | numexpr &amp;gt;= numexpr<br/>
 *               | numexpr = numexpr<br/>
 *               | numexpr != numexpr<br/>
 *               | ( boolexpr )<br/>
 *               | ! boolexpr<br/>
 *               | boolexpr &amp; boolexpr<br/>
 *               | boolexpr | boolexpr<br/>
 *               ;<br/>
 * <br/>
 * Variables are either all upper case letters (e.g., "ABC") or any character apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").
 * <p/>
 <!-- globalinfo-end -->
 *
 * Code example 1:
 * <pre>
 * String expr = "pow(BASE,EXPONENT)*MULT";
 * HashMap symbols = new HashMap();
 * symbols.put("BASE", new Double(2));
 * symbols.put("EXPONENT", new Double(9));
 * symbols.put("MULT", new Double(0.1));
 * double result = MathematicalExpression.evaluate(expr, symbols);
 * System.out.println(expr + " and " + symbols + " = " + result);
 * </pre>
 *
 * Code Example 2 (uses the "ifelse" construct):
 * <pre>
 * String expr = "ifelse(I<0,pow(BASE,I*0.5),pow(BASE,I))";
 * MathematicalExpression.TreeNode tree = MathematicalExpression.parse(expr);
 * HashMap symbols = new HashMap();
 * symbols.put("BASE", new Double(2));
 * for (int i = -10; i <= 10; i++) {
 *   symbols.put("I", new Double(i));
 *   double result = MathematicalExpression.evaluate(expr, symbols);
 *   System.out.println(expr + " and " + symbols + " = " + result);
 * }
 * </pre>
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-env &lt;java.lang.String&gt; (property: environment)
 * &nbsp;&nbsp;&nbsp;The class to use for determining the environment.
 * &nbsp;&nbsp;&nbsp;default: adams.env.Environment
 * </pre>
 * 
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The mathematical expression to evaluate (must evaluate to a double).
 * &nbsp;&nbsp;&nbsp;default: 42
 * </pre>
 * 
 * <pre>-symbol &lt;adams.core.base.BaseString&gt; [-symbol ...] (property: symbols)
 * &nbsp;&nbsp;&nbsp;The symbols to initialize the parser with, key-value pairs: name=value.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4534 $
 */
public class MathematicalExpression
  extends AbstractSymbolEvaluator<Double> {

  /** for serialization. */
  private static final long serialVersionUID = 8014316012335802585L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Evaluates mathematical expressions.\n\n"
    + "The following grammar is used:\n\n"
    + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return
        "expr_list ::= expr_list expr_part | expr_part ;\n"
      + "expr_part ::= expr ;\n"
      + "\n"
      + "numexpr   ::=   ( numexpr )\n"
      + "              |  NUMBER\n"
      + "              | -numexpr\n"
      + "              | constexpr\n"
      + "              | opexpr\n"
      + "              | varexpr\n"
      + "              | funcexpr\n"
      + "              ;\n"
      + "\n"
      + "constexpr ::=   PI\n"
      + "              | E\n"
      + "              ;\n"
      + "\n"
      + "opexpr    ::=   numexpr + numexpr\n"
      + "              | numexpr - numexpr\n"
      + "              | numexpr * numexpr\n"
      + "              | numexpr / numexpr\n"
      + "              | numexpr ^ numexpr\n"
      + "              ;\n"
      + "\n"
      + "varexpr  ::=  VARIABLE ;\n"
      + "\n"
      + "funcexpr ::=    abs ( numexpr )\n"
      + "              | sqrt ( numexpr )\n"
      + "              | log ( numexpr )\n"
      + "              | exp ( numexpr )\n"
      + "              | sin ( numexpr )\n"
      + "              | cos ( numexpr )\n"
      + "              | tan ( numexpr )\n"
      + "              | rint ( numexpr )\n"
      + "              | floor ( numexpr )\n"
      + "              | pow ( numexpr , numexpr )\n"
      + "              | ceil ( numexpr )\n"
      + "              | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )\n"
      + "              | ifmissing ( variable , numexpr (default value if variable is missing) )\n"
      + "              ;\n"
      + "\n"
      + "boolexpr ::=    BOOLEAN\n"
      + "              | true\n"
      + "              | false\n"
      + "              | istrue ( VARIABLE (must represent a boolean string: true|false) )\n"
      + "              | numexpr &lt; numexpr\n"
      + "              | numexpr &lt;= numexpr\n"
      + "              | numexpr &gt; numexpr\n"
      + "              | numexpr &gt;= numexpr\n"
      + "              | numexpr = numexpr\n"
      + "              | numexpr != numexpr\n"
      + "              | ( boolexpr )\n"
      + "              | ! boolexpr\n"
      + "              | boolexpr & boolexpr\n"
      + "              | boolexpr | boolexpr\n"
      + "              ;\n"
      + "\n"
      + "Variables are either all upper case letters (e.g., \"ABC\") or any character "
      + "apart from \"]\" enclosed by \"[\" and \"]\" (e.g., \"[Hello World]\").";
  }

  /**
   * Returns the default expression to use.
   *
   * @return		the default expression
   */
  protected String getDefaultExpression() {
    return "42";
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return "The mathematical expression to evaluate (must evaluate to a double).";
  }

  /**
   * Initializes the symbol.
   *
   * @param name	the name of the symbol
   * @param value	the string representation of the symbol
   * @return		the object representation of the symbol
   */
  protected Object initializeSymbol(String name, String value) {
    Double	result;

    try {
      result = new Double(value);
    }
    catch (Exception e) {
      result = null;
      getSystemErr().println("Failed to parse the value of symbol '" + name + "': " + value);
      getSystemErr().printStackTrace(e);
    }

    return result;
  }

  /**
   * Performs the actual evaluation.
   *
   * @param symbols	the symbols to use
   * @return		the evaluation, or null in case of error
   * @throws Exception	if evaluation fails
   */
  protected Double doEvaluate(HashMap symbols) throws Exception {
    return evaluate(m_Expression, symbols);
  }

  /**
   * Parses and evaluates the given expression.
   * Returns the result of the mathematical expression, based on the given
   * values of the symbols.
   *
   * @param expr	the expression to evaluate
   * @param symbols	the symbol/value mapping
   * @return		the evaluated result
   * @throws Exception	if something goes wrong
   */
  public static double evaluate(String expr, HashMap symbols) throws Exception {
    SymbolFactory 		sf;
    ByteArrayInputStream 	parserInput;
    Parser 			parser;

    sf          = new DefaultSymbolFactory();
    parserInput = new ByteArrayInputStream(expr.getBytes());
    parser      = new Parser(new Scanner(parserInput, sf), sf);
    parser.setSymbols(symbols);
    parser.parse();

    return parser.getResult();
  }

  /**
   * Runs the evaluator from command-line.
   *
   * @param args	the command-line options, use "-help" to list them
   */
  public static void main(String[] args) {
    runEvaluator(MathematicalExpression.class, args);
  }
}
