/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * OptionalFlowExecution.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.condition.test;

import adams.core.io.FlowFile;
import adams.db.DatabaseConnection;
import adams.flow.FlowRunner;

/**
 <!-- globalinfo-start -->
 * Checks whether a specified file exists. If not, then the specified flow is run. After the flow finishes, the check is performed once again, whether the file exists or not.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-file &lt;java.io.File&gt; (property: file)
 *         The file to look for.
 *         default: .
 * </pre>
 *
 * <pre>-headless (property: headless)
 *         If set to true, the actor is run in headless mode without GUI components
 *         .
 * </pre>
 *
 * <pre>-flow &lt;adams.core.io.FlowFile&gt; (property: flowFile)
 *         The flow to run if the file doesn't exist.
 *         default: .
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class OptionalFlowExecution
  extends FileExists {

  /** for serialization. */
  private static final long serialVersionUID = 3297313699016976690L;

  /** the flow to run. */
  protected FlowFile m_FlowFile;

  /** whether the execution is to be headless, i.e., no GUI components. */
  protected boolean m_Headless;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Checks whether a specified file exists. If not, then the specified "
      + "flow is run. After the flow finishes, the check is performed once "
      + "again, whether the file exists or not.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "headless", "headless",
	    false);

    m_OptionManager.add(
	    "flow", "flowFile",
	    new FlowFile("."));
  }

  /**
   * Sets whether the actor is to be run in headless mode, i.e., suppressing
   * GUI components.
   *
   * @param value	if true then GUI components will be suppressed
   */
  public void setHeadless(boolean value) {
    m_Headless = value;
  }

  /**
   * Returns whether the actor is run in headless mode.
   *
   * @return		true if GUI components are suppressed
   */
  public boolean isHeadless() {
    return m_Headless;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String headlessTipText() {
    return "If set to true, the actor is run in headless mode without GUI components.";
  }

  /**
   * Sets the flow to run if necessary.
   *
   * @param value	the flow file
   */
  public void setFlowFile(FlowFile value) {
    m_FlowFile = value;
  }

  /**
   * Returns the flow to run if necessary.
   *
   * @return		the flow file
   */
  public FlowFile getFlowFile() {
    return m_FlowFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String flowFileTipText() {
    return "The flow to run if the file doesn't exist.";
  }

  /**
   * Performs the actual testing of the condition.
   *
   * @return		the test result, null if everything OK, otherwise
   * 			the error message
   */
  protected String performTest() {
    String	result;
    FlowRunner	flow;

    result = null;

    if (!m_File.exists()) {
      flow = new FlowRunner();
      flow.setDatabaseConnection(DatabaseConnection.getSingleton()); // FIXME
      flow.setFile(m_FlowFile);
      flow.setHeadless(isHeadless());
      result = flow.execute();
    }

    if (result == null)
      result = super.performTest();

    return result;
  }
}
