/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * UpdateProperties.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.beans.PropertyDescriptor;
import java.util.Hashtable;

import adams.core.Utils;
import adams.core.VariableName;
import adams.core.base.BaseString;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.ActorUtils;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;
import adams.flow.core.PropertyHelper;
import adams.flow.core.Token;
import adams.flow.transformer.PassThrough;
import adams.gui.goe.PropertyPath;

/**
 <!-- globalinfo-start -->
 * Updates the properties of the sub-actor using the values associated with the specfiied variables. The input&#47;output of the actor are the same as the ones of the sub-actor, since this control actor merely functions as wrapper.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SetProperties
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-property &lt;adams.core.base.BaseString&gt; [-property ...] (property: properties)
 * &nbsp;&nbsp;&nbsp;The properties to update with the values associated with the specified values.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-variable &lt;adams.core.VariableName&gt; [-variable ...] (property: variableNames)
 * &nbsp;&nbsp;&nbsp;The names of the variables to update the properties with.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-sub-actor &lt;adams.flow.core.AbstractActor&gt; (property: subActor)
 * &nbsp;&nbsp;&nbsp;The transformer to update the properties for.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.transformer.PassThrough
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class UpdateProperties
  extends AbstractControlActor
  implements InputConsumer, OutputProducer {

  /** for serialization. */
  private static final long serialVersionUID = -2286932196386647785L;

  /** the key for storing the input token in the backup. */
  public final static String BACKUP_INPUT = "input";

  /** the key for storing the output token in the backup. */
  public final static String BACKUP_OUTPUT = "output";

  /** the current input token. */
  protected transient Token m_InputToken;

  /** the current output token. */
  protected transient Token m_OutputToken;

  /** the property paths. */
  protected BaseString[] m_Properties;

  /** the variables to update the properties with. */
  protected VariableName[] m_VariableNames;

  /** the actor to update the properties foe. */
  protected AbstractActor m_SubActor;

  /** the property descriptors of the properties to update. */
  protected transient PropertyDescriptor[] m_Descriptors;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Updates the properties of the sub-actor using the values associated with "
      + "the specfiied variables. The input/output of the actor are the same "
      + "as the ones of the sub-actor, since this control actor merely functions "
      + "as wrapper.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "property", "properties",
	    new BaseString[0]);

    m_OptionManager.add(
	    "variable", "variableNames",
	    new VariableName[0]);

    m_OptionManager.add(
	    "sub-actor", "subActor",
	    new PassThrough());
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_InputToken  = null;
    m_OutputToken = null;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;

    result = null;

    if (m_Properties.length == 1)
      result = "prop: " + m_Properties[0] + ", var: " + m_VariableNames[0];
    else
      result = m_Properties.length + " props/vars";

    return result;
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_InputToken != null)
      result.put(BACKUP_INPUT, m_InputToken);
    if (m_OutputToken != null)
      result.put(BACKUP_OUTPUT, m_OutputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_INPUT)) {
      m_InputToken = (Token) state.get(BACKUP_INPUT);
      ((InputConsumer) m_SubActor).input(m_InputToken);
      state.remove(BACKUP_INPUT);
    }

    if (state.containsKey(BACKUP_OUTPUT)) {
      m_OutputToken = (Token) state.get(BACKUP_OUTPUT);
      state.remove(BACKUP_OUTPUT);
    }

    super.restoreState(state);
  }

  /**
   * Sets the properties to update.
   *
   * @param value	the properties
   */
  public void setProperties(BaseString[] value) {
    m_Properties    = value;
    m_VariableNames = (VariableName[]) Utils.adjustArray(m_VariableNames, m_Properties.length, new VariableName("unknown"));
    reset();
  }

  /**
   * Returns the properties to update.
   *
   * @return		the properties
   */
  public BaseString[] getProperties() {
    return m_Properties;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String propertiesTipText() {
    return "The properties to update with the values associated with the specified values.";
  }

  /**
   * Sets the variables to use.
   *
   * @param value	the variables
   */
  public void setVariableNames(VariableName[] value) {
    m_VariableNames = value;
    m_Properties    = (BaseString[]) Utils.adjustArray(m_Properties, m_VariableNames.length, new BaseString("subActor.property"));
    reset();
  }

  /**
   * Returns the variables to use.
   *
   * @return		the variables
   */
  public VariableName[] getVariableNames() {
    return m_VariableNames;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String variableNamesTipText() {
    return "The names of the variables to update the properties with.";
  }

  /**
   * Checks the sub-actor before it is set via the setSubActor method.
   * Returns an error message if the actor is not acceptable, null otherwise.
   *
   * @param actor	the actor to check
   * @return		null if accepted, otherwise error message
   */
  protected String checkSubActor(AbstractActor actor) {
    if (!ActorUtils.isTransformer(actor))
      return "Not a transformer: " + actor;

    return null;
  }

  /**
   * Sets the sub actor.
   *
   * @param value	the actor
   */
  public void setSubActor(AbstractActor value) {
    String	msg;

    msg = checkSubActor(value);
    if (msg == null) {
      m_SubActor = value;
      reset();
      updateParent();
    }
    else {
      throw new IllegalArgumentException(msg);
    }
  }

  /**
   * Returns the sub-actor.
   *
   * @return		the actor
   */
  public AbstractActor getSubActor() {
    return m_SubActor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String subActorTipText() {
    return "The transformer to update the properties for.";
  }

  /**
   * Returns the size of the group.
   *
   * @return		always 1
   */
  public int size() {
    return 1;
  }

  /**
   * Returns the actor at the given position.
   *
   * @param index	the position
   * @return		the actor
   */
  public AbstractActor get(int index) {
    if (index == 0)
      return m_SubActor;
    else
      throw new IllegalArgumentException("Illegal index: " + index);
  }

  /**
   * Sets the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to set at this position
   */
  public void set(int index, AbstractActor actor) {
    if (index == 0)
      setSubActor(actor);
    else
      throw new IllegalArgumentException("Illegal index: " + index);
  }

  /**
   * Returns the index of the actor.
   *
   * @param actor	the name of the actor to look for
   * @return		the index of -1 if not found
   */
  public int indexOf(String actor) {
    if (m_SubActor.getName().equals(actor))
      return 0;
    else
      return -1;
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(false, ActorExecution.SEQUENTIAL, true);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return ((InputConsumer) m_SubActor).accepts();
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   * @see		#m_InputToken
   */
  public void input(Token token) {
    m_InputToken = token;
    ((InputConsumer) m_SubActor).input(m_InputToken);
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return ((OutputProducer) m_SubActor).generates();
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;
    Class	cls;
    int		i;

    result = super.setUp();

    if (result == null) {
      m_Descriptors = new PropertyDescriptor[m_Properties.length];
      for (i = 0; i < m_Properties.length; i++) {
	m_Descriptors[i] = PropertyPath.getPropertyDescriptor(m_SubActor, m_Properties[i].getValue());
	if (m_Descriptors[i] == null) {
	  result = "Cannot find property '" + m_Properties[i] + "' in sub actor!";
	}
	else {
	  cls = m_Descriptors[i].getReadMethod().getReturnType();
	  if (cls.isArray())
	    result = "Property '" + m_Properties[i] + "' is an array!";
	}
	if (result != null)
	  break;
      }
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    String	msg;
    int		i;
    Object	value;

    result = null;

    for (i = 0; i < m_Properties.length; i++) {
      try {
	value = PropertyHelper.convertVariableValue(
	    m_Descriptors[i],
	    getVariables().get(m_VariableNames[i].getValue()));
	PropertyPath.setValue(
	    m_SubActor,
	    m_Properties[i].stringValue(),
	    value);
      }
      catch (Exception e) {
	msg = "Failed to set property '" + m_Properties[i] + "': ";
	getSystemErr().println(msg);
	getSystemErr().printStackTrace(e);
	if (result == null)
	  result = msg + e;
	else
	  result += "\n" + msg + e;
      }
    }

    if (result == null)
      result = m_SubActor.execute();

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String execute() {
    if (m_Skip)
      m_OutputToken = m_InputToken;

    return super.execute();
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return ((OutputProducer) m_SubActor).hasPendingOutput() || (m_OutputToken != null);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    m_InputToken = null;

    if (m_OutputToken != null)
      return m_OutputToken;
    else
      return ((OutputProducer) m_SubActor).output();
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    m_InputToken  = null;
    m_OutputToken = null;

    super.wrapUp();
  }
}
