/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * HistoryDisplay.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JSplitPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.flow.core.Token;
import adams.gui.core.AbstractNamedHistoryPanel.HistoryEntrySelectionEvent;
import adams.gui.core.AbstractNamedHistoryPanel.HistoryEntrySelectionListener;
import adams.gui.core.BasePanel;
import adams.gui.core.BufferHistoryPanel;
import adams.gui.core.GUIHelper;
import adams.gui.core.TextEditorPanel;

/**
 <!-- globalinfo-start -->
 * Actor that outputs any object that arrives at its input port via the 'toString()' method in a separate 'history' entry.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Object<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: HistoryDisplay
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 640
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 480
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-font &lt;java.awt.Font&gt; (property: font)
 * &nbsp;&nbsp;&nbsp;The font of the dialog.
 * &nbsp;&nbsp;&nbsp;default: Monospaced-PLAIN-12
 * </pre>
 *
 * <pre>-caret-at-start (property: caretAtStart)
 * &nbsp;&nbsp;&nbsp;If set to true, then the caret will be positioned by default at the start
 * &nbsp;&nbsp;&nbsp;and not the end (can be changed in dialog: View -&gt; Position caret at start
 * &nbsp;&nbsp;&nbsp;).
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class HistoryDisplay
  extends AbstractTextualDisplay {

  /** for serialization. */
  private static final long serialVersionUID = 3365817040968234289L;

  /**
   * Represents a panel with a history on the left and the displayed text
   * on the right.
   *
   * @author FracPete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public static class HistorySplitPanel
    extends BasePanel {

    /** for serialization. */
    private static final long serialVersionUID = 5121061351955687610L;

    /** the owning HistoryDisplay component. */
    protected HistoryDisplay m_Owner;

    /** the split pane for the components. */
    protected JSplitPane m_SplitPane;

    /** the history panel. */
    protected BufferHistoryPanel m_History;

    /** the actual text area. */
    protected TextEditorPanel m_TextPanel;

    /** the format for the dates. */
    protected SimpleDateFormat m_Format;

    /**
     * Initializes the split pane.
     *
     * @param owner		the owning TextDisplay
     */
    public HistorySplitPanel(HistoryDisplay owner) {
      super(new BorderLayout());

      m_Owner   = owner;
      m_Format  = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

      m_SplitPane = new JSplitPane();
      add(m_SplitPane, BorderLayout.CENTER);

      m_TextPanel = new TextEditorPanel();
      m_TextPanel.setTextFont(owner.getFont());
      m_TextPanel.setEditable(false);
      m_SplitPane.setBottomComponent(m_TextPanel);

      m_History = new BufferHistoryPanel();
      m_History.setTextArea(m_TextPanel.getTextArea());
      m_SplitPane.setTopComponent(m_History);

      m_SplitPane.setResizeWeight(0.1);
      m_SplitPane.setDividerLocation(150);
    }

    /**
     * Returns the owner of this history panel.
     *
     * @return		the owning TextDisplay component
     */
    public HistoryDisplay getOwner() {
      return m_Owner;
    }

    /**
     * Returns the underlying text panel.
     *
     * @return		the text editor
     */
    public TextEditorPanel getTextPanel() {
      return m_TextPanel;
    }

    /**
     * Removes all entries.
     */
    public void clear() {
      m_History.clear();
      m_TextPanel.clear();
    }

    /**
     * Returns the number of results.
     *
     * @return		the number of results
     */
    public int count() {
      return m_History.count();
    }

    /**
     * Adds the given text.
     *
     * @param result	the text to add
     */
    public void addResult(String result) {
      addResult(new StringBuffer(result));
    }

    /**
     * Returns the underlying history panel.
     *
     * @return		the panel
     */
    public BufferHistoryPanel getHistory() {
      return m_History;
    }

    /**
     * Adds the given text.
     *
     * @param result	the text to add
     */
    public synchronized void addResult(StringBuffer result) {
      String	baseID;
      String	id;
      int	count;

      // determine a unique ID
      synchronized(m_Format) {
	baseID = m_Format.format(new Date());
      }
      id     = baseID;
      count  = 1;
      while (m_History.hasEntry(id)) {
	count++;
	id = baseID + " (" + count + ")";
      }

      // add result
      m_History.addEntry(id, result);

      // select this entry immediately
      m_History.setSelectedIndex(count() - 1);
    }
  }

  /** the history panel. */
  protected HistorySplitPanel m_HistoryPanel;

  /** the print menu item. */
  protected JMenuItem m_MenuItemFilePrint;

  /** the copy menu item. */
  protected JMenuItem m_MenuItemEditCopy;

  /** the select all menu item. */
  protected JMenuItem m_MenuItemEditSelectAll;

  /** the font menu item. */
  protected JMenuItem m_MenuItemViewFont;

  /** the "caret position" menu item. */
  protected JCheckBoxMenuItem m_MenuItemViewCaret;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Actor that outputs any object that arrives at its input port via "
      + "the 'toString()' method in a separate 'history' entry.";
  }

  /** whether to position the caret by default at the start or at the end. */
  protected boolean m_CaretAtStart;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "caret-at-start", "caretAtStart",
	    false);
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 640;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 480;
  }

  /**
   * Sets whether to position the caret at the start or at the end (default).
   *
   * @param value	if true then the caret will be positioned at start
   */
  public void setCaretAtStart(boolean value) {
    m_CaretAtStart = value;
    reset();
  }

  /**
   * Returns whether the caret is positioned at the start instead of the end.
   *
   * @return		true if caret positioned at start
   */
  public boolean isCaretAtStart() {
    return m_CaretAtStart;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String caretAtStartTipText() {
    return
        "If set to true, then the caret will be positioned by default at the "
      + "start and not the end (can be changed in dialog: View -> Position caret at start).";
  }

  /**
   * Clears the content of the panel.
   */
  public void clearPanel() {
    m_HistoryPanel.clear();
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    m_HistoryPanel = new HistorySplitPanel(this);
    m_HistoryPanel.getHistory().setCaretAtStart(m_CaretAtStart);
    m_HistoryPanel.getHistory().addHistoryEntrySelectionListener(new HistoryEntrySelectionListener() {
      public void historyEntrySelected(HistoryEntrySelectionEvent e) {
	updateMenu();
      }
    });

    return m_HistoryPanel;
  }

  /**
   * Creates the "File" menu.
   *
   * @return		the generated menu
   */
  protected JMenu createFileMenu() {
    JMenu	result;
    JMenuItem	menuitem;
    int		pos;

    result = super.createFileMenu();

    // File/Print
    menuitem = new JMenuItem("Print...");
    menuitem.setMnemonic('P');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed P"));
    menuitem.setIcon(GUIHelper.getIcon("print.gif"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_HistoryPanel.getTextPanel().printText();
      }
    });
    pos = indexOfMenuItem(result, m_MenuItemFileClose);
    result.insertSeparator(pos);
    result.insert(menuitem, pos);
    m_MenuItemFilePrint = menuitem;

    return result;
  }

  /**
   * Creates the "Edit" menu.
   *
   * @return		the menu
   */
  protected JMenu createEditMenu() {
    JMenu			result;
    JMenuItem			menuitem;
    final TextEditorPanel	fPanel;

    fPanel = m_HistoryPanel.getTextPanel();

    // Edit
    result = new JMenu("Edit");
    result.setMnemonic('E');
    result.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	updateMenu();
      }
    });

    // Edit/Copy
    menuitem = new JMenuItem("Copy", GUIHelper.getIcon("copy.gif"));
    menuitem.setMnemonic('C');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed C"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	fPanel.copy();
      }
    });
    result.add(menuitem);
    m_MenuItemEditCopy = menuitem;

    // Edit/Select all
    menuitem = new JMenuItem("Select all", GUIHelper.getEmptyIcon());
    menuitem.setMnemonic('S');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed A"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	fPanel.selectAll();
      }
    });
    result.addSeparator();
    result.add(menuitem);
    m_MenuItemEditSelectAll = menuitem;

    return result;
  }

  /**
   * Creates the "Edit" menu.
   *
   * @return		the menu
   */
  protected JMenu createViewMenu() {
    JMenu	result;
    JMenuItem	menuitem;

    // View
    result = new JMenu("View");
    result.setMnemonic('V');
    result.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	updateMenu();
      }
    });

    // View/Font
    menuitem = new JMenuItem("Choose font...");
    result.add(menuitem);
    menuitem.setIcon(GUIHelper.getIcon("font.png"));
    menuitem.setMnemonic('f');
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_HistoryPanel.getTextPanel().selectFont();
      }
    });
    m_MenuItemViewFont = menuitem;

    // View/Caret
    menuitem = new JCheckBoxMenuItem("Position caret at start");
    result.addSeparator();
    result.add(menuitem);
    menuitem.setMnemonic('s');
    menuitem.setSelected(m_CaretAtStart);
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_HistoryPanel.getHistory().setCaretAtStart(m_MenuItemViewCaret.isSelected());
      }
    });
    m_MenuItemViewCaret = (JCheckBoxMenuItem) menuitem;

    return result;
  }

  /**
   * Assembles the menu bar.
   *
   * @return		the menu bar
   */
  protected JMenuBar createMenuBar() {
    JMenuBar	result;

    result = super.createMenuBar();
    result.add(createEditMenu());
    result.add(createViewMenu());

    return result;
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateMenu() {
    TextEditorPanel	panel;

    if (m_MenuBar == null)
      return;

    super.updateMenu();

    panel = m_HistoryPanel.getTextPanel();

    // File
    m_MenuItemFileClear.setEnabled(m_HistoryPanel.count() > 0);

    // Edit
    m_MenuItemEditCopy.setEnabled(panel.canCopy());
  }

  /**
   * Whether "clear" is supported and shows up in the menu.
   *
   * @return		always true
   */
  protected boolean supportsClear() {
    return true;
  }

  /**
   * Clears the display.
   */
  protected void clear() {
    m_HistoryPanel.clear();
  }

  /**
   * Returns the text to save.
   *
   * @return		the text, null if no text available
   */
  public String supplyText() {
    String	result;
    int		index;

    result = null;

    index  = m_HistoryPanel.getHistory().getSelectedIndex();
    if (index != -1)
      result = m_HistoryPanel.getHistory().getEntry(index).toString();

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.Object.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Object.class};
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  protected void display(Token token) {
    synchronized(m_HistoryPanel) {
      m_HistoryPanel.addResult(token.getPayload().toString());
    }
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    if (m_HistoryPanel != null)
      m_HistoryPanel.clear();

    super.cleanUpGUI();
  }
}
