/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MakeContainer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.source;

import java.util.Enumeration;

import adams.core.Utils;
import adams.core.base.BaseString;
import adams.flow.container.AbstractContainer;
import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.AbstractActor;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Assembles a container with data obtained from global actors.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * <p/>
 * Container information:<br/>
 * - adams.flow.container.SequencePlotterContainer: PlotName, X, Y
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: MakeContainer
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-global-actor &lt;adams.flow.core.GlobalActorReference&gt; [-global-actor ...] (property: globalActors)
 * &nbsp;&nbsp;&nbsp;The global actors to obtain the data from.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-value-name &lt;adams.core.base.BaseString&gt; [-value-name ...] (property: valueNames)
 * &nbsp;&nbsp;&nbsp;The names to use for storing the values in the container:PlotName, X, Y.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-container-class &lt;adams.flow.container.AbstractContainer&gt; (property: containerClass)
 * &nbsp;&nbsp;&nbsp;The container class to generate (full class name).
 * &nbsp;&nbsp;&nbsp;default: adams.flow.container.SequencePlotterContainer
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class MakeContainer
  extends AbstractSource {

  /** for serialization. */
  private static final long serialVersionUID = -132045002653940359L;

  /** the global actors to retrieve the data from. */
  protected GlobalActorReference[] m_GlobalActors;

  /** the names under which to store the data. */
  protected BaseString[] m_ValueNames;

  /** the container class to create. */
  protected AbstractContainer m_ContainerClass;

  /** the output token to broadcast. */
  protected Token m_OutputToken;

  /** the names of the items for the current container. */
  protected String m_ContainerValues;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Assembles a container with data obtained from global actors.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "global-actor", "globalActors",
	    new GlobalActorReference[0]);

    m_OptionManager.add(
	    "value-name", "valueNames",
	    new BaseString[0]);

    m_OptionManager.add(
	    "container-class", "containerClass",
	    new SequencePlotterContainer());
  }

  /**
   * Resets the members.
   */
  protected void reset() {
    super.reset();

    m_OutputToken     = null;
    m_ContainerValues = null;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("containerClass");
    if (variable != null)
      return variable;
    else
      return m_ContainerClass.getClass().getName();
  }

  /**
   * Sets the names of the global actors to obtain the data from.
   *
   * @param value	the names
   */
  public void setGlobalActors(GlobalActorReference[] value) {
    m_GlobalActors = value;
    m_ValueNames   = (BaseString[]) Utils.adjustArray(m_ValueNames, m_GlobalActors.length, new BaseString("unknown"));
    reset();
  }

  /**
   * Returns the names of the global actors to get the data form.
   *
   * @return 		the names
   */
  public GlobalActorReference[] getGlobalActors() {
    return m_GlobalActors;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String globalActorsTipText() {
    return "The global actors to obtain the data from.";
  }

  /**
   * Sets the names of the values in the container.
   *
   * @param value	the names
   */
  public void setValueNames(BaseString[] value) {
    m_ValueNames   = value;
    m_GlobalActors = (GlobalActorReference[]) Utils.adjustArray(m_GlobalActors, m_ValueNames.length, new GlobalActorReference("unknown"));
    reset();
  }

  /**
   * Returns the names to store the values under in the container.
   *
   * @return 		the names
   */
  public BaseString[] getValueNames() {
    return m_ValueNames;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String valueNamesTipText() {
    String	values;

    values = getContainerValues();
    if (values != null)
      values = values.replace(",", ", ");

    return
        "The names to use for storing the values in the container"
      + ((values == null) ? "" : ":" + values) + ".";
  }

  /**
   * Sets the class name of the container to create.
   *
   * @param value	the class name
   */
  public void setContainerClass(AbstractContainer value) {
    m_ContainerClass = value;
    reset();
  }

  /**
   * Returns the class name of the container to create.
   *
   * @return 		the class name
   */
  public AbstractContainer getContainerClass() {
    return m_ContainerClass;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String containerClassTipText() {
    return "The container class to generate (full class name).";
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.container.SequencePlotterContainer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    AbstractContainer	cont;

    cont = m_ContainerClass.getClone();
    if (cont != null)
      return new Class[]{cont.getClass()};
    else
      return new Class[]{Object.class};
  }

  /**
   * Returns a comma-separated list of names that the container accepts.
   *
   * @return		the list of names, null if no container class name
   * 			or invalid classname specified
   */
  protected synchronized String getContainerValues() {
    Enumeration<String>	names;

    if (m_ContainerValues == null) {
      m_ContainerValues = "";
      names             = m_ContainerClass.names();
      while (names.hasMoreElements()) {
	if (m_ContainerValues.length() > 0)
	  m_ContainerValues += ",";
	m_ContainerValues += names.nextElement();
      }
    }

    return m_ContainerValues;
  }

  /**
   * Returns the value obtained from the global actor.
   *
   * @param name	the name of the global actor to get the value from
   * @return		the obtained value
   */
  protected Object getValue(GlobalActorReference name) {
    GlobalSource	global;
    Object		result;
    AbstractActor	actor;
    Token		token;

    result = null;

    global = new GlobalSource();
    global.setGlobalName(name);
    global.setParent(getParent());
    if (global.setUp() == null) {
      global.execute();
      token = global.output();
      if (token != null) {
	result = token.getPayload();
	actor  = global.getGlobalActor();
	global.wrapUp();
	global.cleanUp();
	// fix wrapUp/cleanUp of global actor, e.g., re-registering variable listeners
	if (actor != null)
	  actor.setUp();
      }
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    Object		value;
    AbstractContainer	cont;
    int			i;

    result = null;

    cont = getContainerClass().getClone();
    for (i = 0; i < m_GlobalActors.length; i++) {
      value = getValue(m_GlobalActors[i]);
      cont.setValue(m_ValueNames[i].getValue(), value);
    }
    if (cont.isValid())
      m_OutputToken = new Token(cont);

    return result;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = m_OutputToken;
    m_OutputToken = null;

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_OutputToken != null);
  }
}
