/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GlobalActors.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.standalone;


import java.util.HashSet;

import adams.flow.control.AbstractDirector;
import adams.flow.control.MutableControlActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;

/**
 <!-- globalinfo-start -->
 * Container for actors that need to be accessed globally via their name.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: GlobalActors
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseString&gt; [-annotation ...] (property: annotations)
 *         The annotations to attach to this actor.
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-progress (property: showProgress)
 *         If set to true, progress information will be output to stdout ('.').
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor [options]&gt; [-actor ...] (property: actors)
 *         All the actors that are to be accessed globally via their name.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class GlobalActors
  extends MutableControlActor {

  /** for serialization. */
  private static final long serialVersionUID = -5282103315016094476L;

  /**
   * Dummy director.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public static class GlobalActorsDirector
    extends AbstractDirector {

    /** for serialization. */
    private static final long serialVersionUID = -7581710637774405432L;

    /**
     * Executes the group of actors.
     *
     * @return		always null
     */
    public String execute() {
      return null;
    }
  }

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Container for actors that need to be accessed globally via their name.";
  }

  /**
   * Returns an instance of a director.
   *
   * @return		the director
   */
  protected AbstractDirector newDirector() {
    return new GlobalActorsDirector();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorsTipText() {
    return "All the actors that are to be accessed globally via their name.";
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(true, ActorExecution.PARALLEL, false);
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    int		i;
    String	result;

    result = super.setUp();

    if (result == null) {
      // check whether everything is correctly setup
      for (i = 0; i < size(); i++) {
	if (get(i).getSkip())
	  continue;
	result = get(i).setUp();
	if (result != null)
	  break;
      }
    }

    return result;
  }

  /**
   * Checks whether the names of the contained actors are unique or not.
   *
   * @return		null if names unique, otherwise offending pair
   */
  protected String checkNames() {
    String		result;
    HashSet<String>	names;
    int			i;

    result = null;

    names = new HashSet<String>();
    for (i = 0; i < size(); i++) {
      if (names.contains(get(i).getName())) {
	result = "Actor '" + get(i).getFullName() + "' has duplicate name '" + get(i).getName() + "'!";
	break;
      }
      else {
	names.add(get(i).getName());
      }
    }

    return result;
  }

  /**
   * Checks the names for uniqueness.
   *
   * @return		null if everything is fine, otherwise the offending
   * 			connection
   */
  public String check() {
    return checkNames();
  }

  /**
   * Does nothing.
   *
   * @return		null
   */
  protected String doExecute() {
    return null;
  }

  /**
   * Finishes up the execution.
   */
  public void wrapUp() {
    int		i;

    for (i = 0; i < size(); i++)
      get(i).wrapUp();

    super.wrapUp();
  }
}
