/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GUIHelper.java
 * Copyright (C) 2008-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.core;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.image.BufferedImage;
import java.io.File;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JInternalFrame;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.UIManager;

import adams.core.Properties;
import adams.core.Utils;
import adams.core.management.OS;
import adams.env.Environment;
import adams.env.GUIHelperDefinition;
import adams.gui.application.Child;

/**
 * A little helper class for GUI related stuff.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class GUIHelper {

  /** the name of the props file. */
  public final static String FILENAME = "GUIHelper.props";

  /** the empty icon name. */
  public final static String EMPTY_ICON = "empty.gif";

  /** the mnemonic character indicator. */
  public final static char MNEMONIC_INDICATOR = '_';

  /** the properties. */
  protected static Properties m_Properties;

  /** the screen boundaries. */
  protected static Rectangle m_ScreenBoundaries;

  /**
   * Initializes the properties if necessary.
   */
  protected static synchronized void initializeProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(GUIHelperDefinition.KEY);
  }

  /**
   * Checks whether the image is available.
   *
   * @param name	the name of the image (filename without path but with
   * 			extension)
   * @return		true if image exists
   */
  public static boolean hasImageFile(String name) {
    return (getImageFilename(name) != null);
  }

  /**
   * Adds the path of the images directory to the name of the image.
   *
   * @param name	the name of the image to add the path to
   * @return		the full path of the image
   */
  public static String getImageFilename(String name) {
    String	result;
    String[]	dirs;
    int		i;
    URL		url;

    result = null;

    dirs = getString("ImagesDirectory", "adams/gui/images/").split(",");
    for (i = 0; i < dirs.length; i++) {
      if (!dirs[i].endsWith("/"))
	dirs[i] += "/";
      try {
	url = ClassLoader.getSystemClassLoader().getResource(dirs[i] + name);
	if (url != null) {
	  result = dirs[i] + name;
	  break;
	}
      }
      catch (Exception e) {
	// ignored
      }
    }

    return result;
  }

  /**
   * Returns an ImageIcon for the given class.
   *
   * @param cls		the class to get the icon for (gif, png or jpg)
   * @return		the ImageIcon or null if none found
   */
  public static ImageIcon getIcon(Class cls) {
    if (hasImageFile(cls.getName() + ".gif"))
      return getIcon(cls.getName() + ".gif");
    else if (hasImageFile(cls.getName() + ".png"))
      return getIcon(cls.getName() + ".png");
    else if (hasImageFile(cls.getName() + ".jpg"))
      return getIcon(cls.getName() + ".jpg");
    else
      return null;
  }

  /**
   * Returns an ImageIcon from the given name.
   *
   * @param name	the filename without path
   * @return		the ImageIcon or null if not available
   */
  public static ImageIcon getIcon(String name) {
    String	filename;

    filename = getImageFilename(name);
    if (filename != null)
      return new ImageIcon(ClassLoader.getSystemClassLoader().getResource(filename));
    else
      return null;
  }

  /**
   * Returns an ImageIcon from the given name.
   *
   * @param filename	the filename
   * @return		the ImageIcon or null if not available
   */
  public static ImageIcon getExternalIcon(String filename) {
    ImageIcon	result;

    try {
      result = new ImageIcon(ClassLoader.getSystemClassLoader().getResource(filename));
    }
    catch (Exception e) {
      result = null;
    }

    return result;
  }

  /**
   * Returns the ImageIcon for the empty icon.
   *
   * @return		the ImageIcon
   */
  public static ImageIcon getEmptyIcon() {
    return getIcon(EMPTY_ICON);
  }

  /**
   * Returns an ImageIcon of the logo (large image).
   *
   * @return		the logo or null if none available
   */
  public static ImageIcon getLogoImage() {
    ImageIcon	result;
    String	filename;

    result = null;

    filename = getString("LogoImage", "");
    if (filename.length() != 0)
      result = getIcon(filename);

    return result;
  }

  /**
   * Returns an ImageIcon of the logo (icon sized image).
   *
   * @return		the logo or null if none available
   */
  public static ImageIcon getLogoIcon() {
    ImageIcon	result;
    String	filename;

    result = null;

    filename = getString("LogoIcon", "");
    if (filename.length() != 0)
      result = getIcon(filename);

    return result;
  }

  /**
   * Sets size stored in the props file.
   *
   * @param window	the frame to work on
   * @param c		the component to use for lookup in the props file;
   * 			null can be used to bypass the lookup in the props file
   */
  public static void setSize(Window window, Component c) {
    int		width;
    int		height;

    initializeProperties();

    // custom size?
    if (c != null) {
      if (m_Properties.getBoolean(c.getClass().getName() + ".pack", false))
	window.pack();

      width = m_Properties.getInteger(c.getClass().getName() + ".width", window.getWidth());
      if (width == -1)
	width = window.getGraphicsConfiguration().getBounds().width;

      height = m_Properties.getInteger(c.getClass().getName() + ".height", window.getHeight());
      if (height == -1)
	height = window.getGraphicsConfiguration().getBounds().height;

      window.setSize(width, height);
      window.validate();
    }
  }

  /**
   * Sets size and location stored in the props file.
   *
   * @param window	the frame to work on
   * @see		#setSizeAndLocation(Window, Component)
   */
  public static void setSizeAndLocation(Window window) {
    setSizeAndLocation(window, null);
  }

  /**
   * Sets size and location stored in the props file.
   *
   * @param window	the frame to work on
   * @param c		the component to use for lookup in the props file;
   * 			null can be used to bypass the lookup in the props file
   * 			(in that case, "top" and "left" are initialized with 0)
   * @see		#setSizeAndLocation(Window, int, int, Component)
   */
  public static void setSizeAndLocation(Window window, Component c) {
    int		left;
    int		top;

    initializeProperties();

    // determine size first
    setSize(window, c);

    // custom location
    if (c != null) {
      left = m_Properties.getInteger(c.getClass().getName() + ".left", window.getX());
      if (left == -1)
	left = (window.getGraphicsConfiguration().getBounds().width - window.getWidth()) / 2;
      else if (left == -2)
	left = window.getGraphicsConfiguration().getBounds().width - left;
      if (left < 0)
	left = 0;
    }
    else {
      left = 0;
    }

    if (c != null) {
      top = m_Properties.getInteger(c.getClass().getName() + ".top", window.getY());
      if (top == -1)
	top = (window.getGraphicsConfiguration().getBounds().height - window.getHeight()) / 2;
      else if (top == -2)
	top = window.getGraphicsConfiguration().getBounds().height - top;
      if (top < 0)
	top = 0;
    }
    else {
      top = 0;
    }

    setSizeAndLocation(window, top, left, c);
  }

  /**
   * Sets size and location stored in the props file.
   * <p/>
   * Takes the following parameters from the props file into account as well:
   * MaxPercentHeight, ScreenBorder.Bottom
   *
   * @param window	the frame to work on
   * @param top		the y position
   * @param left	the x position
   * @see		#setSizeAndLocation(Window, int, int, Component)
   */
  public static void setSizeAndLocation(Window window, int top, int left) {
    setSizeAndLocation(window, top, left, null);
  }

  /**
   * Sets size and location stored in the props file.
   * <p/>
   * Takes the following parameters from the props file into account as well:
   * MaxPercentHeight, ScreenBorder.Bottom
   *
   * @param window	the frame to work on
   * @param top		the y position
   * @param left	the x position
   * @param c		the component to use for lookup in the props file;
   * 			null can be used to bypass the lookup in the props file
   */
  public static void setSizeAndLocation(Window window, int top, int left, Component c) {
    int		width;
    int		height;
    Rectangle	screen;

    initializeProperties();

    // custom size?
    setSize(window, c);
    adjustSize(window);

    // position
    height = window.getHeight();
    width  = window.getWidth();
    screen = getScreenBounds(window);
    if (left + width > screen.width)
      left = screen.width - width;
    if (top + height > screen.height)
      top = screen.height - height;

    window.setLocation(left, top);
  }

  /**
   * Returns the actual screen real estate bounds according to
   * ScreenBorder.Top/Left/Bottom/Right in the props file.
   *
   * <pre>
   * +----------------------------+  physical screen
   * |                            |
   * |   +--------------------+   |
   * |   |                    |   |
   * |   |  available screen  |   |
   * |   |                    |   |
   * |   |                    |   |
   * |   +--------------------+   |
   * |                            |
   * |                            |
   * +----------------------------+
   * </pre>
   *
   * @param window	the window to get the graphics config from
   * @return		the "inner" rectangle where we can display stuff
   */
  public static synchronized Rectangle getScreenBounds(Window window) {
    int		height;
    int 	width;
    int		top;
    int		left;
    int		bottom;
    int		right;

    if (m_ScreenBoundaries == null) {
      initializeProperties();

      top     = m_Properties.getInteger("ScreenBorder.Top", 0);
      left    = m_Properties.getInteger("ScreenBorder.Left", 0);
      bottom  = m_Properties.getInteger("ScreenBorder.Bottom", 0);
      right   = m_Properties.getInteger("ScreenBorder.Right", 0);
      height  = window.getGraphicsConfiguration().getBounds().height - top - bottom;
      width   = window.getGraphicsConfiguration().getBounds().width - left - right;

      m_ScreenBoundaries = new Rectangle(left, top, width, height);
    }

    return m_ScreenBoundaries;
  }

  /**
   * Adjusts the size of the window, that it fits onto the screen.
   *
   * @param window	the window to adjust
   * @see		#getScreenBounds(Window)
   */
  public static void adjustSize(Window window) {
    Rectangle	screen;
    double	percHeight;
    double	percWidth;
    int		height;
    int		width;

    screen = getScreenBounds(window);
    height = window.getHeight();
    width  = window.getWidth();

    percHeight = m_Properties.getDouble("MaxWindowHeight", 0.95);
    if ((percHeight <= 0) || (percHeight > 1))
      percHeight = 0.95;
    percWidth = m_Properties.getDouble("MaxWindowWidth", 0.95);
    if ((percWidth <= 0) || (percWidth > 1))
      percWidth = 0.95;

    if (height > (double) screen.height * percHeight)
      height = (int) ((double) screen.height * percHeight);
    if (width > (double) screen.width * percWidth)
      width = (int) ((double) screen.width * percWidth);

    window.setSize(width, height);
    window.validate();
  }

  /**
   * Returns the startup script, if any, for the given component.
   *
   * @param c		the component to look for a startup script for
   * @return		the script file or null if none listed or none-existing
   */
  public static File getStartupScript(Component c) {
    File	result;
    String	key;
    String	script;

    initializeProperties();

    result = null;

    key = c.getClass().getName() + ".script";
    if (m_Properties.hasKey(key)) {
      script = m_Properties.getString(key);
      result = new File(script);
      if (!result.exists()) {
	System.err.println(
	    "Startup script '" + script
	    + "' listed for component '" + c.getClass().getName()
	    + "' does not exist - ignored!");
	result = null;
      }
    }

    return result;
  }

  /**
   * Returns the string value listed in the props file, or the default value
   * if not found.
   *
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static String getString(String key, String defValue) {
    initializeProperties();

    return m_Properties.getProperty(key, defValue);
  }

  /**
   * Returns the string value listed in the props file, or the default value
   * if not found. The key used is this: classname + "." + key.
   *
   * @param cls		the class to retrieve the key for
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static String getString(Class cls, String key, String defValue) {
    return getString(cls.getName() + "." + key, defValue);
  }

  /**
   * Returns the integer value listed in the props file, or the default value
   * if not found.
   *
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static Integer getInteger(String key, Integer defValue) {
    initializeProperties();

    return m_Properties.getInteger(key, defValue);
  }

  /**
   * Returns the integer value listed in the props file, or the default value
   * if not found. The key used is this: classname + "." + key.
   *
   * @param cls		the class to retrieve the key for
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static Integer getInteger(Class cls, String key, Integer defValue) {
    return getInteger(cls.getName() + "." + key, defValue);
  }

  /**
   * Returns the double value listed in the props file, or the default value
   * if not found.
   *
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static Double getDouble(String key, Double defValue) {
    initializeProperties();

    return m_Properties.getDouble(key, defValue);
  }

  /**
   * Returns the double value listed in the props file, or the default value
   * if not found. The key used is this: classname + "." + key.
   *
   * @param cls		the class to retrieve the key for
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static Double getDouble(Class cls, String key, Double defValue) {
    return getDouble(cls.getName() + "." + key, defValue);
  }

  /**
   * Returns the boolean value listed in the props file, or the default value
   * if not found.
   *
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static Boolean getBoolean(String key, Boolean defValue) {
    initializeProperties();

    return m_Properties.getBoolean(key, defValue);
  }

  /**
   * Returns the boolean value listed in the props file, or the default value
   * if not found. The key used is this: classname + "." + key.
   *
   * @param cls		the class to retrieve the key for
   * @param key		the key of the property
   * @param defValue	the default value to return if property is not stored
   * 			in props file
   * @return		the value
   */
  public static Boolean getBoolean(Class cls, String key, Boolean defValue) {
    return getBoolean(cls.getName() + "." + key, defValue);
  }

  /**
   * Tries to determine the parent this panel is part of.
   *
   * @param cont	the container to get the parent for
   * @param parentClass	the class of the parent to obtain
   * @return		the parent if one exists or null if not
   */
  public static Object getParent(Container cont, Class parentClass) {
    Container	result;
    Container	parent;

    result = null;

    parent = cont;
    while (parent != null) {
      if (parentClass.isInstance(parent)) {
	result = parent;
	break;
      }
      else {
	parent = parent.getParent();
      }
    }

    return result;
  }

  /**
   * Tries to determine the frame the container is part of.
   *
   * @param cont	the container to get the frame for
   * @return		the parent frame if one exists or null if not
   */
  public static Frame getParentFrame(Container cont) {
    return (Frame) getParent(cont, Frame.class);
  }

  /**
   * Tries to determine the dialog this panel is part of.
   *
   * @param cont	the container to get the dialog for
   * @return		the parent dialog if one exists or null if not
   */
  public static Dialog getParentDialog(Container cont) {
    return (Dialog) getParent(cont, Dialog.class);
  }

  /**
   * Tries to determine the component this panel is part of in this order:
   * 1. dialog, 2. child, 3. frame.
   *
   * @param comp	the component to get the parent component for, must
   * 			be a container actually
   * @return		the parent component if one exists or null if not
   * @see		#getParentDialog(Container)
   * @see		#getParentChild(Container)
   * @see		#getParentFrame(Container)
   */
  public static Component getParentComponent(Component comp) {
    Component	result;
    Container	cont;

    if (comp instanceof Container)
      cont = (Container) comp;
    else
      return null;

    result = getParentDialog(cont);
    if (result == null)
      result = (Component) getParentChild(cont);
    if (result == null)
      result = getParentFrame(cont);

    return result;
  }

  /**
   * Tries to determine the internal frame this panel is part of.
   *
   * @param cont		the container to start with
   * @return		the parent internal frame if one exists or null if not
   */
  public static JInternalFrame getParentInternalFrame(Container cont) {
    return (JInternalFrame) getParent(cont, JInternalFrame.class);
  }

  /**
   * Tries to determine the child window/frame this panel is part of.
   *
   * @param cont	the container to get the child window/frame for
   * @return		the parent child window/frame if one exists or null if not
   */
  public static Child getParentChild(Container cont) {
    return (Child) getParent(cont, Child.class);
  }

  /**
   * Copies the given string to the system's clipboard.
   *
   * @param s		the string to copy
   */
  public static void copyToClipboard(String s) {
    TransferableString 	selection;
    Clipboard 		clipboard;

    selection = new TransferableString(s);
    clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
    clipboard.setContents(selection, null);
  }

  /**
   * Copies the given image to the system's clipboard.
   *
   * @param img		the image to copy
   */
  public static void copyToClipboard(BufferedImage img) {
    TransferableImage 	selection;
    Clipboard 		clipboard;

    selection = new TransferableImage(img);
    clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
    clipboard.setContents(selection, null);
  }

  /**
   * Checks whether a string can be obtained from the clipboard.
   *
   * @return		true if string can be obtained, false if not available
   */
  public static boolean canPasteFromClipboard() {
    Clipboard 		clipboard;
    boolean		result;

    try {
      clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
      result    = clipboard.isDataFlavorAvailable(DataFlavor.stringFlavor);
    }
    catch (Exception e) {
      result = false;
    }

    return result;
  }

  /**
   * Obtains a string from the clipboard.
   *
   * @return		the obtained string, null if not available
   */
  public static String pasteFromClipboard() {
    Clipboard 		clipboard;
    String		result;
    Transferable	content;

    result = null;

    try {
      clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
      content   = clipboard.getContents(null);
      if ((content != null) && (content.isDataFlavorSupported(DataFlavor.stringFlavor)))
	result = (String) content.getTransferData(DataFlavor.stringFlavor);
    }
    catch (Exception e) {
      result = null;
    }

    return result;
  }

  /**
   * Obtains a setup string from the clipboard. This command can be spread
   * across several lines, each line (apart from last one) ending with a "\".
   * If that is the case, then the lines are collapsed into a single line.
   *
   * @return		the obtained string, null if not available
   */
  public static String pasteSetupFromClipboard() {
    StringBuilder	result;
    String[]		parts;
    Vector<String>	lines;
    String		line;
    int			i;
    boolean		formatOK;

    result = new StringBuilder(pasteFromClipboard());

    if (result != null) {
      parts = result.toString().replaceAll("\r", "").split("\n");
      if (parts.length > 1) {
	// preprocess string
	lines = new Vector<String>();
	for (i = 0; i < parts.length; i++) {
	  line = parts[i].trim();
	  if (line.length() == 0)
	    continue;
	  lines.add(line);
	}

	// check format:
	// all lines apart from last one must have a trailing backslash
	formatOK = true;
	for (i = 0; i < lines.size(); i++) {
	  if (i < lines.size() - 1)
	    formatOK = lines.get(i).endsWith("\\");
	  else
	    formatOK = !lines.get(i).endsWith("\\");
	  if (!formatOK)
	    break;
	}

	// collapse the command
	if (formatOK) {
	  result = new StringBuilder();
	  for (i = 0; i < lines.size(); i++) {
	    if (i > 0)
	      result.append(" ");
	    line = lines.get(i);
	    if (i < lines.size() - 1)
	      line = line.substring(0, lines.get(i).length() - 1).trim();
	    result.append(line);
	  }
	}
      }
    }

    return result.toString();
  }

  /**
   * Returns the look'n'feel classname from the props file.
   *
   * @return		the classname
   */
  public static String getLookAndFeel() {
    return getString("LookAndFeel", UIManager.getCrossPlatformLookAndFeelClassName());
  }

  /**
   * Installs the specified look'n'feel.
   *
   * @param classname	the classname of the look'n'feel to install
   * @return		true if successfully installed
   */
  public static boolean setLookAndFeel(String classname) {
    boolean	result;

    try {
      UIManager.setLookAndFeel(classname);
      result = true;
    }
    catch (Exception e) {
      result = false;
      System.err.println("Can't set look & feel:" + e);
    }

    return result;
  }

  /**
   * Suggests mnemonics for the given labels.
   *
   * @param labels	the labels to set the mnemonics for
   * @return		the mnemonics
   */
  public static char[] getMnemonics(String[] labels) {
    char[]					result;
    String      				allowed;
    String[]					strLabels;
    Hashtable<Character,HashSet<Integer>>	charIndices;
    int						i;
    int						n;
    char					ch;
    Enumeration<Character>			enm;
    HashSet<Integer>				processedIndices;
    int						numLabels;

    result = new char[labels.length];

    // a-z and 0-9 minus (O)K, (C)ancel and (M)ore
    allowed = "abdefghijklnpqrstuvwxyz0123456789";

    // determine letters and numbers per label
    strLabels = new String[labels.length];
    numLabels = 0;
    for (i = 0; i < labels.length; i++) {
      strLabels[i] = labels[i].toLowerCase().replaceAll("[^a-z0-9]", "");
      if (strLabels[i].length() == 0)
	strLabels[i] = null;
      else
	numLabels++;
    }

    // determine counts of characters across the labels
    charIndices = new Hashtable<Character,HashSet<Integer>>();
    for (i = 0; i < allowed.length(); i++) {
      ch = allowed.charAt(i);
      for (n = 0; n < strLabels.length; n++) {
	if (strLabels[n] == null)
	  continue;
	if (strLabels[n].indexOf(ch) > -1) {
	  if (!charIndices.containsKey(ch))
	    charIndices.put(ch, new HashSet<Integer>());
	  charIndices.get(ch).add(n);
	}
      }
    }

    // set the mnemonics for the labels which are the most unique
    i                = 0;
    processedIndices = new HashSet<Integer>();
    do {
      i++;
      enm = charIndices.keys();
      while (enm.hasMoreElements()) {
	ch = enm.nextElement();
	if (charIndices.get(ch).size() == i) {
	  for (Integer index: charIndices.get(ch)) {
	    if (index < 0)
	      continue;
	    if (strLabels[index] == null)
	      continue;
	    result[index]    = ch;
	    strLabels[index] = null;
	  }

	  // flag indices as 'processed'
	  processedIndices.addAll(charIndices.get(ch));
	  processedIndices.remove(-1);
	}
      }
    }
    while (processedIndices.size() != numLabels);

    return result;
  }

  /**
   * Checks the caption whether an underscore "_" is present to indicate
   * that the following character is to act as mnemonic.
   *
   * @param caption	the caption to analyze
   * @return		true if an underscore is present
   * @see		#MNEMONIC_INDICATOR
   */
  public static boolean hasMnemonic(String caption) {
    return (caption.indexOf(MNEMONIC_INDICATOR) > -1);
  }

  /**
   * Returns the mnemonic for this caption, preceded by an underscore "_".
   *
   * @param caption	the caption to extract
   * @return		the extracted mnemonic, \0 if none available
   * @see		#MNEMONIC_INDICATOR
   */
  public static char getMnemonic(String caption) {
    int		pos;

    pos = caption.indexOf(MNEMONIC_INDICATOR);
    if ((pos > -1) && (pos < caption.length() - 1))
      return caption.charAt(pos + 1);
    else
      return '\0';
  }

  /**
   * Removes the mnemonic indicator in this caption.
   *
   * @param caption	the caption to process
   * @return		the processed caption
   * @see		#MNEMONIC_INDICATOR
   */
  public static String stripMnemonic(String caption) {
    return caption.replace("" + MNEMONIC_INDICATOR, "");
  }

  /**
   * Displays an error message with the default title "Error".
   *
   * @param parent	the parent, to make the dialog modal; can be null
   * @param msg		the error message to display
   */
  public static void showErrorMessage(Component parent, String msg) {
    showErrorMessage(parent, msg, "Error");
  }

  /**
   * Displays an error message.
   *
   * @param parent	the parent, to make the dialog modal; can be null
   * @param msg		the error message to display
   * @param title	the title of the error message
   */
  public static void showErrorMessage(Component parent, String msg, String title) {
    JOptionPane.showMessageDialog(
	parent,
	msg,
	title,
	JOptionPane.ERROR_MESSAGE);
  }

  /**
   * Displays an information message with the default title "Information".
   *
   * @param parent	the parent, to make the dialog modal; can be null
   * @param msg		the information message to display
   */
  public static void showInformationMessage(Component parent, String msg) {
    showInformationMessage(parent, msg, "Information");
  }

  /**
   * Displays an information message.
   *
   * @param parent	the parent, to make the dialog modal; can be null
   * @param msg		the information message to display
   * @param title	the title of the information message
   */
  public static void showInformationMessage(Component parent, String msg, String title) {
    JOptionPane.showMessageDialog(
	parent,
	msg,
	title,
	JOptionPane.INFORMATION_MESSAGE);
  }

  /**
   * Parses the tool tip and breaks it up into multiple lines if longer
   * than width characters.
   *
   * @param text	the tiptext to parse
   * @param width	the maximum width
   * @return		the processed tiptext
   */
  public static String processTipText(String text, int width) {
    String	result;
    String[]	lines;

    result = text;
    if (result.length() > width) {
      lines  = Utils.breakUp(result, width);
      result = Utils.flatten(lines, "<br>");
    }
    result = "<html>" + result + "</html>";

    return result;
  }

  /**
   * Processes the keystrokes. For Macs, "ctrl/control" is replaced by "meta".
   * Also, if no "ctrl/control/alt" is present, a "meta" is prefix.
   * All other platforms simply return the string.
   *
   * @param keystroke	the keystroke string to process
   * @return		the (potentially) processed keystroke definition
   */
  public static String processKeyStroke(String keystroke) {
    String	result;

    result = keystroke;

    if (OS.isMac()) {
      result = result.replace("ctrl", "meta").replace("control", "meta");
      if ((result.indexOf("meta") == -1) && (result.indexOf("alt") == -1))
	result = "meta " + result;
    }

    return result;
  }

  /**
   * Creates a keystroke from the string.
   *
   * @param keystroke	the keystroke string to turn into a
   * @return		the generated keystroke
   * @see		#processKeyStroke(String)
   */
  public static KeyStroke getKeyStroke(String keystroke) {
    return KeyStroke.getKeyStroke(processKeyStroke(keystroke));
  }
}
