/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractAxisModel.java
 * Copyright (C) 2008-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core.axis;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Stack;
import java.util.Vector;

import adams.gui.visualization.core.AxisPanel;

/**
 * An abstract class of an axis model.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractAxisModel
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = -3950212023344727427L;

  /** the owning axis. */
  protected AxisPanel m_Parent;

  /** the minimum. */
  protected double m_Minimum;

  /** the maximum. */
  protected double m_Maximum;

  /** the actual minimum to display (including the bottom margin). */
  protected double m_ActualMinimum;

  /** the actual maximum to display (including the top margin). */
  protected double m_ActualMaximum;

  /** the number of ticks to display. */
  protected int m_NumTicks;

  /** every nth value to display. */
  protected int m_NthValueToShow;

  /** the top margin. */
  protected double m_MarginTop;

  /** the bottom margin. */
  protected double m_MarginBottom;

  /** indicates whether the layout has been validated. */
  protected boolean m_Validated;

  /** the format for outputting the values (SimpleDateFormat or DecimalFormat). */
  protected Formatter m_Formatter;

  /** a customer formatter to use. */
  protected Formatter m_CustomerFormatter;

  /** the zooms. */
  protected ZoomHandler m_ZoomHandler;

  /** a pixel offset due to panning. */
  protected int m_PixelOffset;

  /** stack of pixel offsets for the zooms. */
  protected Stack<Integer> m_PixelOffsets;

  /**
   * Initializes the model.
   */
  public AbstractAxisModel() {
    super();

    initialize();
  }

  /**
   * Initializes the member variables.
   */
  protected void initialize() {
    m_Parent            = null;
    m_Minimum           = 0.0;
    m_Maximum           = 1.0;
    m_MarginTop         = 0.0;
    m_MarginBottom      = 0.0;
    m_PixelOffset       = 0;
    m_Validated         = false;
    m_NumTicks          = 20;
    m_NthValueToShow    = 5;
    m_Formatter         = Formatter.getDecimalFormatter(getDefaultNumberFormat());
    m_CustomerFormatter = null;
    m_ZoomHandler       = new ZoomHandler();
    m_PixelOffsets      = new Stack<Integer>();
  }

  /**
   * Sets the owning axis panel.
   *
   * @param value	the axis panel
   */
  public void setParent(AxisPanel value) {
    m_Parent = value;
  }

  /**
   * Returns the owning axis panel.
   *
   * @return		the axis panel
   */
  public AxisPanel getParent() {
    return m_Parent;
  }

  /**
   * Returns the display name of this model.
   *
   * @return		the display name
   */
  public abstract String getDisplayName();

  /**
   * Checks whether the data range can be handled by the model.
   *
   * @param min		the minimum value
   * @param max		the maximum value
   * @return		true if the data can be handled
   */
  public abstract boolean canHandle(double min, double max);

  /**
   * Sets the minimum to display on the axis.
   *
   * @param value	the minimum value
   */
  public void setMinimum(double value) {
    m_Minimum = value;
    invalidate();
    update();
  }

  /**
   * Returns the currently set minimum on the axis.
   *
   * @return		the minimum value
   */
  public double getMinimum() {
    return m_Minimum;
  }

  /**
   * Sets the maximum to display on the axis.
   *
   * @param value	the maximum value
   */
  public void setMaximum(double value) {
    m_Maximum = value;
    invalidate();
    update();
  }

  /**
   * Returns the currently set maximum on the axis.
   *
   * @return		the minimum value
   */
  public double getMaximum() {
    return m_Maximum;
  }

  /**
   * Sets the top margin factor (>= 0.0).
   *
   * @param value	the top margin
   */
  public void setTopMargin(double value) {
    if (value >= 0) {
      m_MarginTop = value;
      invalidate();
      update();
    }
    else {
      System.err.println(
	  "Top margin factor must be at least 0.0 (provided: " + value + ")!");
    }
  }

  /**
   * Returns the currently set top margin factor (>= 0.0).
   *
   * @return		the top margin
   */
  public double getTopMargin() {
    return m_MarginTop;
  }

  /**
   * Sets the bottom margin factor (>= 0.0).
   *
   * @param value	the bottom margin
   */
  public void setBottomMargin(double value) {
    if (value >= 0) {
      m_MarginBottom = value;
      invalidate();
      update();
    }
    else {
      System.err.println(
	  "Bottom margin factor must be at least 0.0 (provided: " + value + ")!");
    }
  }

  /**
   * Returns the currently set bottom margin factor (>= 0.0).
   *
   * @return		the bottom margin
   */
  public double getBottomMargin() {
    return m_MarginBottom;
  }

  /**
   * Sets the pixel offset due to panning.
   *
   * @param value	the offset
   */
  public void setPixelOffset(int value) {
    m_PixelOffset = value;
    invalidate();
    update();
  }

  /**
   * Returns the current pixel offset.
   *
   * @return		the offset
   */
  public int getPixelOffset() {
    return m_PixelOffset;
  }

  /**
   * Returns the default number format.
   *
   * @return		the default format
   */
  public String getDefaultNumberFormat() {
    return "0.00E0;-0.00E0";
  }

  /**
   * Sets the pattern used for displaying the numbers on the axis.
   *
   * @param value	the value to use
   */
  public void setNumberFormat(String value) {
    m_Formatter.applyPattern(value);
    update();
  }

  /**
   * Returns the pattern used for displaying the numbers on the axis.
   *
   * @return		the pattern
   */
  public String getNumberFormat() {
    return m_Formatter.toPattern();
  }

  /**
   * Returns whether a custom formatter is in use.
   *
   * @return		true if a custom formatter is used
   */
  public boolean hasCustomFormatter() {
    return (m_CustomerFormatter != null);
  }

  /**
   * Sets the custom formatter to use. Use null to unset the formatter.
   *
   * @param value	the custom formatter to use
   */
  public void setCustomFormatter(Formatter value) {
    m_CustomerFormatter = value;
  }

  /**
   * Returns the current custom formatter, can be null if none set.
   *
   * @return		the custom formatter
   */
  public Formatter getCustomFormatter() {
    return m_CustomerFormatter;
  }

  /**
   * Returns the formatter to use for parsing/formatting.
   *
   * @return		the formatter to use
   * @see		#m_Formatter
   * @see		#m_CustomerFormatter
   */
  protected Formatter getActualFormatter() {
    if (hasCustomFormatter())
      return m_CustomerFormatter;
    else
      return m_Formatter;
  }

  /**
   * Sets the count of ticks a value is shown, i.e., "3" means every third tick:
   * 1, 4, 7, ...
   *
   * @param value	the count
   */
  public void setNthValueToShow(int value) {
    if (value >= 0) {
      m_NthValueToShow = value;
      update();
    }
    else {
      System.err.println("'n-th value to show' must be >=0, provided: " + value);
    }
  }

  /**
   * Returns the count of ticks a value is shown, i.e., "3" means every third
   * tick: 1, 4, 7, ...
   *
   * @return		the count
   */
  public int getNthValueToShow() {
    return m_NthValueToShow;
  }

  /**
   * Sets the number of ticks to display along the axis.
   *
   * @param value	the number of ticks
   */
  public void setNumTicks(int value) {
    if (value > 0) {
      m_NumTicks = value;
      update();
    }
    else {
      System.err.println("Number of ticks must be >0, provided: " + value);
    }
  }

  /**
   * Returns the number of ticks currently displayed.
   *
   * @return		the number of ticks
   */
  public int getNumTicks() {
    return m_NumTicks;
  }

  /**
   * Returns the minimum difference that must exist between min/max in order
   * to allow zooming.
   * 
   * @return		the minimum difference
   */
  protected double getMinZoomDifference() {
    return 10E-8;
  }
  
  /**
   * Checks whether we can still zoom in.
   *
   * @param min		the minimum of the zoom
   * @param max		the maximum of the zoom
   * @return		true if zoom is possible
   */
  public boolean canZoom(double min, double max) {
    return (Math.abs(max - min) > getMinZoomDifference());
  }

  /**
   * Adds the zoom to its internal list and updates the axis.
   *
   * @param min		the minimum of the zoom
   * @param max		the maximum of the zoom
   */
  public void pushZoom(double min, double max) {
    m_ZoomHandler.push(min, max);
    m_PixelOffsets.push(m_PixelOffset);
    m_PixelOffset = 0;

    invalidate();
    update();
  }

  /**
   * Removes the latest zoom, if available.
   */
  public void popZoom() {
    if (isZoomed()) {
      m_ZoomHandler.pop();
      m_PixelOffset = m_PixelOffsets.pop();
      invalidate();
      update();
    }
  }

  /**
   * Returns true if the axis is currently zoomed.
   *
   * @return		true if a zoom is in place
   */
  public boolean isZoomed() {
    return m_ZoomHandler.isZoomed();
  }

  /**
   * Removes all zooms.
   */
  public void clearZoom() {
    if (isZoomed()) {
      m_ZoomHandler.clear();
      m_PixelOffset = 0;
      m_PixelOffsets.clear();
      invalidate();
      update();
    }
  }

  /**
   * Returns the ticks of this axis.
   *
   * @return		the current ticks to display
   */
  public Vector<Tick> getTicks() {
    Vector<Tick>	result;
    int			i;
    int			pos;
    String		label;
    double		incValue;
    double		value;
    HashSet<String>	labels;

    result   = new Vector<Tick>();
    incValue = (m_ActualMaximum - m_ActualMinimum) / ((double) m_NumTicks);
    labels   = new HashSet<String>();

    for (i = 0; i < m_NumTicks + 1; i++) {
      value = m_ActualMinimum + i * incValue;
      pos   = valueToPos(value);
      label = valueToDisplay(posToValue(pos));
      if (labels.contains(label))
	label = " ";
      if (label.equals("NaN"))
	label = " ";
      if ((m_NthValueToShow == 0) || (i % m_NthValueToShow != 0))
	label = null;
      result.add(new Tick(pos, label));
      if (label != null)
	labels.add(label);
    }

    return result;
  }

  /**
   * Returns the display string of the value for the tooltip, for instance.
   *
   * @param value	the value to turn into string
   * @return		the display string
   */
  public abstract String valueToDisplay(double value);

  /**
   * Returns the value from the display string.
   *
   * @param display	the string to turn into a value
   * @return		the value
   */
  public abstract double displayToValue(String display);

  /**
   * Returns the position on the axis for the given value.
   *
   * @param value	the value to get the position for
   * @return		the corresponding position
   */
  public abstract int valueToPos(double value);

  /**
   * Returns the value for the given position on the axis.
   *
   * @param pos	the position to get the corresponding value for
   * @return		the corresponding value
   */
  public abstract double posToValue(int pos);

  /**
   * Invalidates the current setup, calculations necessary for margins, etc.
   */
  public void invalidate() {
    m_Validated = false;
  }

  /**
   * calculates the top and bottom margin if necessary.
   */
  public void validate() {
    double	range;
    double	min;
    double	max;
    double	offset;
    double	size;

    if (m_Validated)
      return;

    if (m_ZoomHandler.isZoomed()) {
      min = m_ZoomHandler.peek().getMinimum();
      max = m_ZoomHandler.peek().getMaximum();
    }
    else {
      min = m_Minimum;
      max = m_Maximum;
    }

    if (getParent().getLength() == 0)
      size = 1;
    else
      size = getParent().getLength();
    range           = Math.abs(max - min);
    offset          = range / size * (double) m_PixelOffset;
    m_ActualMinimum = min - range * m_MarginBottom - offset;
    m_ActualMaximum = max + range * m_MarginTop    - offset;

    m_Validated = true;
  }

  /**
   * Obtains the necessary values from the given model and updates itself.
   *
   * @param model	the model to get the parameters from
   */
  public void assign(AbstractAxisModel model) {
    m_Parent            = model.m_Parent;
    m_Minimum           = model.m_Minimum;
    m_Maximum           = model.m_Maximum;
    m_MarginTop         = model.m_MarginTop;
    m_MarginBottom      = model.m_MarginBottom;
    m_ZoomHandler       = (ZoomHandler) model.m_ZoomHandler.getClone();
    m_PixelOffsets      = (Stack<Integer>) model.m_PixelOffsets.clone();
    m_PixelOffset       = model.m_PixelOffset;
    m_CustomerFormatter = model.m_CustomerFormatter;
    m_NthValueToShow    = model.m_NthValueToShow;
    m_NumTicks          = model.m_NumTicks;

    invalidate();
    update();
  }

  /**
   * Forces the panel to repaint itself.
   */
  public void update() {
    validate();
    if (m_Parent != null) {
      m_Parent.repaint();
      m_Parent.notifyChangeListeners();
    }
  }

  /**
   * Returns a string representation of the model.
   *
   * @return		 a string representation
   */
  public String toString() {
    String	result;

    result  = getClass().getName() + ": ";
    result += "min=" + getMinimum() + ", ";
    result += "max=" + getMaximum() + ", ";
    result += "topMargin=" + getTopMargin() + ", ";
    result += "bottomMargin=" + getBottomMargin() + ", ";
    result += "format=" + getNumberFormat();

    return result;
  }
}