/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractXYSequencePaintlet.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.sequence;

import java.awt.Color;

import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.AbstractPaintlet;
import adams.gui.visualization.core.PaintablePanel;

/**
 * Abstract superclass for paintlets for X-Y sequences.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractXYSequencePaintlet
  extends AbstractPaintlet {

  /** for serialization. */
  private static final long serialVersionUID = 1570802737796372715L;

  /** the hit detector to use. */
  protected AbstractXYSequencePointHitDetector m_HitDetector;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_HitDetector = newHitDetector();
  }

  /**
   * Returns a new instance of the hit detector to use.
   *
   * @return		the hit detector
   */
  public abstract AbstractXYSequencePointHitDetector newHitDetector();

  /**
   * Sets the spectrum panel to use, null to disable painting.
   *
   * @param value	the panel to paint on
   */
  public void setPanel(PaintablePanel value) {
    super.setPanel(value);
    m_HitDetector.setOwner(getSequencePanel());
  }

  /**
   * Returns the XY sequence panel currently in use.
   *
   * @return		the panel in use
   */
  public XYSequencePanel getSequencePanel() {
    return (XYSequencePanel) m_Panel;
  }

  /**
   * Returns when this paintlet is to be executed.
   *
   * @return		when this paintlet is to be executed
   */
  public PaintMoment getPaintMoment() {
    return PaintMoment.PAINT;
  }

  /**
   * Returns the color for the data with the given index.
   *
   * @param index	the index of the spectrum
   * @return		the color for the spectrum
   */
  public Color getColor(int index) {
    return getSequencePanel().getContainerManager().get(index).getColor();
  }

  /**
   * Returns the hit detector to use for this paintlet.
   *
   * @return		the detector
   */
  public AbstractXYSequencePointHitDetector getHitDetector() {
    return m_HitDetector;
  }
}
