/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetTest.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.core.io;

import java.io.File;

import junit.framework.Test;
import junit.framework.TestSuite;
import adams.env.Environment;
import adams.test.AbstractTestHelper;
import adams.test.AdamsTestCase;
import adams.test.TestHelper;
import adams.test.TmpFile;

/**
 * Tests the adams.core.io.SpreadSheet class. Run from commandline with: <p/>
 * java adams.core.io.SpreadSheet
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class SpreadSheetTest
  extends AdamsTestCase {

  /**
   * Initializes the test.
   *
   * @param name	the name of the test
   */
  public SpreadSheetTest(String name) {
    super(name);
  }

  /**
   * Returns the test helper class to use.
   *
   * @return		the helper class instance
   */
  protected AbstractTestHelper newTestHelper() {
    return new TestHelper(this, "adams/core/io/data");
  }

  /**
   * Called by JUnit before each test method.
   *
   * @throws Exception if an error occurs
   */
  protected void setUp() throws Exception {
    super.setUp();

    m_TestHelper.copyResourceToTmp("sample.csv");
    m_TestHelper.copyResourceToTmp("sample2.csv");
  }

  /**
   * Called by JUnit after each test method.
   *
   * @throws Exception	if tear-down fails
   */
  protected void tearDown() throws Exception {
    m_TestHelper.deleteFileFromTmp("sample.csv");
    m_TestHelper.deleteFileFromTmp("sample2.csv");

    super.tearDown();
  }

  /**
   * Tests the loading of a sample spreadsheet.
   */
  public void testRead() {
    SpreadSheet sheet = new CsvSpreadSheetReader().read(new TmpFile("sample.csv").getAbsolutePath());
    assertEquals("# of columns differ", 3, sheet.getColumnCount());
    assertEquals("# of rows differ", 16, sheet.getRowCount());
  }

  /**
   * Tests the cell positions.
   */
  public void testCellPositions() {
    int row = 0;
    int col = 0;
    String pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "A1", pos);

    row = 1;
    col = 2;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "C2", pos);

    row = 1;
    col = 25;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "Z2", pos);

    row = 1;
    col = 26;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "AA2", pos);

    row = 2;
    col = 51;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "AZ3", pos);

    row = 2;
    col = 52;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "BA3", pos);

    row = 2;
    col = 701;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "ZZ3", pos);

    row = 2;
    col = 702;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "AAA3", pos);

    row = 20;
    col = 900;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "AHQ21", pos);

    row = 23;
    col = 1000;
    pos = SpreadSheet.getCellPosition(row, col);
    assertEquals("position differs", "ALM24", pos);
  }

  /**
   * Regression test.
   */
  public void testRegression() {
    String	regression;
    SpreadSheet sheet;
    File	output;

    sheet  = new CsvSpreadSheetReader().read(new TmpFile("sample.csv").getAbsolutePath());
    output = new TmpFile("sample_out.csv");
    new CsvSpreadSheetWriter().write(sheet, output.getAbsolutePath());
    regression = m_Regression.compare(new File[]{output.getAbsoluteFile()});
    assertNull("Output differs:\n" + regression, regression);
  }

  /**
   * Returns a test suite.
   *
   * @return		the test suite
   */
  public static Test suite() {
    return new TestSuite(SpreadSheetTest.class);
  }

  /**
   * Runs the test from commandline.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    Environment.setEnvironmentClass(Environment.class);
    runTest(suite());
  }
}
