/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractFilterTestCase.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.data.filter;

import java.util.Vector;

import adams.core.CleanUpHandler;
import adams.core.Destroyable;
import adams.core.management.ProcessUtils;
import adams.core.option.OptionHandler;
import adams.core.option.OptionUtils;
import adams.data.AbstractDataProcessorTestCase;
import adams.data.container.DataContainer;
import adams.test.AbstractTestHelper;
import adams.test.TestHelper;

/**
 * Ancestor for filter test cases.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @param <A> the type of algorithm to use
 * @param <D> the type of data to filter
 */
public abstract class AbstractFilterTestCase<A extends AbstractFilter, D extends DataContainer>
  extends AbstractDataProcessorTestCase<A, D, D> {

  /**
   * Constructs the test case. Called by subclasses.
   *
   * @param name 	the name of the test
   */
  public AbstractFilterTestCase(String name) {
    super(name);
  }

  /**
   * Returns the test helper class to use.
   *
   * @return		the helper class instance
   */
  protected AbstractTestHelper newTestHelper() {
    return new TestHelper(this, "adams/data/filter/data");
  }

  /**
   * Returns the ignored line indices to use in the regression test.
   *
   * @return		the setups
   */
  protected int[] getRegressionIgnoredLineIndices() {
    return new int[0];
  }

  /**
   * Processes the input data and returns the processed data.
   *
   * @param data	the data to work on
   * @param scheme	the scheme to process the data with
   * @return		the processed data
   */
  protected D process(D data, A scheme) {
    return (D) scheme.filter(data);
  }

  /**
   * Tests whether the input data gets changed when processing it in parallel.
   */
  public void testMultiProcess() {
    D				data;
    D				backup;
    Vector<DataContainer>	processed;
    int				i;
    String[]			input;
    A[]				setups;
    Vector<AbstractFilter>	current;
    String[]			props;
    int				numFilters;
    int				n;

    if (!m_MultiProcessEnabled)
      return;

    numFilters = ProcessUtils.getAvailableProcessors() + 1;
    input      = getRegressionInputFiles();
    setups     = getRegressionSetups();
    props      = getRegressionConnections();
    assertEquals("Number of files and setups differ!", input.length, setups.length);
    if (props != null) {
      assertEquals("Number of files and connection setups differ!", input.length, props.length);
    }
    else {
      props = new String[input.length];
      for (i = 0; i < props.length; i++)
	props[i] = getDatabasePropertiesFile();
    }

    // process data
    for (i = 0; i < input.length; i++) {
      // connect to correct database
      reconnect(props[i]);

      data = load(input[i]);
      assertNotNull("Could not load data for regression test from " + input[i], data);
      backup = (D) data.getClone();

      current = new Vector<AbstractFilter>();
      for (n = 0; n < numFilters; n++) {
	current.add((A) OptionUtils.shallowCopy((OptionHandler) setups[i], false));
	assertNotNull("Failed to create copy of algorithm: " + OptionUtils.getCommandLine(setups[i]), current.get(n));
      }

      processed = AbstractFilter.filter(current, data);
      assertNotNull("Failed to process data?", processed);

      assertTrue("Input got changed", data.equals(backup));
      for (n = 1; n < numFilters; n++)
	assertTrue("Output differs for #" + (n+1), processed.get(0).equals(processed.get(n)));

      if (setups[i] instanceof Destroyable) {
	for (n = 0; n < numFilters; n++)
	  ((Destroyable) current.get(n)).destroy();
      }
    }

    // remove output, clean up scheme
    for (i = 0; i < setups.length; i++) {
      if (setups[i] instanceof Destroyable)
	((Destroyable) setups[i]).destroy();
      else if (setups[i] instanceof CleanUpHandler)
	((CleanUpHandler) setups[i]).cleanUp();
    }

    // connect to default database
    m_Properties = null;
    getDatabaseProperties();
  }
}
