/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ConnectionParameters.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.db;

import java.io.Serializable;
import java.util.Enumeration;
import java.util.Vector;

import adams.core.CloneHandler;
import adams.core.base.BasePassword;

/**
 * Container class for connection information.
 *
 *  @author  fracpete (fracpete at waikato dot ac dot nz)
 *  @version $Revision: 4584 $
 */
public class ConnectionParameters
  implements Serializable, Comparable<ConnectionParameters>, CloneHandler<ConnectionParameters> {

  /** for serialization. */
  private static final long serialVersionUID = -1414581492377334939L;

  /** the class parameter. */
  public final static String PARAM_CLASS = "Class";

  /** the URL parameter. */
  public final static String PARAM_URL = "URL";

  /** the user parameter. */
  public final static String PARAM_USER = "User";

  /** the password parameter. */
  public final static String PARAM_PASSWORD = "Password";

  /** the debug level parameter. */
  public final static String PARAM_DEBUGLEVEL = "DebugLevel";

  /** the connect on startup parameter. */
  public final static String PARAM_CONNECTONSTARTUP = "ConnectOnStartup";

  /** the URL. */
  protected String m_URL;

  /** the user. */
  protected String m_User;

  /** the password. */
  protected BasePassword m_Password;

  /** the debug level. */
  protected int m_DebugLevel;

  /** whether to connect on startup. */
  protected boolean m_ConnectOnStartUp;

  /**
   * Initializes the container.
   *
   * @param url		the JDBC URL
   * @param user	the database user
   * @param password	the database password
   * @param debugLevel	the debug level
   * @param connect	whether to connect on startup
   */
  public ConnectionParameters() {
    super();
    initialize();
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    m_URL              = "";
    m_User             = "";
    m_Password         = new BasePassword();
    m_DebugLevel       = 0;
    m_ConnectOnStartUp = false;
  }

  /**
   * Returns the URL.
   *
   * @return		the URL
   */
  public String getURL() {
    return m_URL;
  }

  /**
   * Returns the user.
   *
   * @return		the user
   */
  public String getUser() {
    return m_User;
  }

  /**
   * Returns the password.
   *
   * @return		the password
   */
  public BasePassword getPassword() {
    return m_Password;
  }

  /**
   * Returns the debug level.
   *
   * @return		the debug level
   */
  public int getDebugLevel() {
    return m_DebugLevel;
  }

  /**
   * Returns whether to connect on startup.
   *
   * @return		true if to connect on startup
   */
  public boolean getConnectOnStartUp() {
    return m_ConnectOnStartUp;
  }

  /**
   * Returns the available parameter keys.
   *
   * @return		the parameter keys
   */
  public Enumeration<String> parameters() {
    Vector<String>	result;

    result = new Vector<String>();

    result.add(PARAM_CLASS);
    result.add(PARAM_URL);
    result.add(PARAM_USER);
    result.add(PARAM_PASSWORD);
    result.add(PARAM_DEBUGLEVEL);
    result.add(PARAM_CONNECTONSTARTUP);

    return result.elements();
  }

  /**
   * Returns the parameter for the specified key.
   *
   * @param key		the key of the parameter to retrieve
   * @return		the associated value, null if not available
   */
  public String getParameter(String key) {
    if (key.equals(PARAM_CLASS))
      return getClass().getName();
    if (key.equals(PARAM_URL))
      return m_URL;
    if (key.equals(PARAM_USER))
      return m_User;
    if (key.equals(PARAM_PASSWORD))
      return m_Password.stringValue();
    if (key.equals(PARAM_DEBUGLEVEL))
      return "" + m_DebugLevel;
    if (key.equals(PARAM_CONNECTONSTARTUP))
      return "" + m_ConnectOnStartUp;

    return null;
  }

  /**
   * Returns the parameter for the specified key.
   *
   * @param key		the key of the parameter to retrieve
   * @return		the associated value
   */
  public void setParameter(String key, String value) {
    if (key.equals(PARAM_CLASS))
      ;  // ignored
    if (key.equals(PARAM_URL))
      m_URL = value;
    else if (key.equals(PARAM_USER))
      m_User = value;
    else if (key.equals(PARAM_PASSWORD))
      m_Password = new BasePassword(value);
    else if (key.equals(PARAM_DEBUGLEVEL))
      m_DebugLevel = Integer.parseInt(value);
    else if (key.equals(PARAM_CONNECTONSTARTUP))
      m_ConnectOnStartUp = Boolean.parseBoolean(value);
  }

  /**
   * Returns a new (empty) instance of a ConnectionParameters object.
   *
   * @return		the empty instance
   */
  protected ConnectionParameters newInstance() {
    return new ConnectionParameters();
  }

  /**
   * Returns a copy of ifself.
   *
   * @return		the copy
   */
  public ConnectionParameters getClone() {
    ConnectionParameters	result;
    Enumeration<String>		keys;
    String			key;

    result = newInstance();
    keys   = parameters();
    while (keys.hasMoreElements()) {
      key = keys.nextElement();
      result.setParameter(key, getParameter(key));
    }

    return result;
  }

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   *
   * @param   o the object to be compared.
   * @return  a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   * @throws ClassCastException if the specified object's type prevents it
   *         from being compared to this object.
   */
  public int compareTo(ConnectionParameters o) {
    int				result;
    Enumeration<String>		keys;
    String			key;
    Object			oThis;
    Object			oOther;

    if (o == null)
      return 1;

    keys   = parameters();
    result = 0;
    while ((result == 0) && keys.hasMoreElements()) {
      key    = keys.nextElement();
      oThis  = getParameter(key);
      oOther = o.getParameter(key);
      if (oOther == null) {
	result = 1;
      }
      else {
	if (oThis instanceof Comparable)
	  result = ((Comparable) oThis).compareTo(oOther);
      }
    }

    return result;
  }

  /**
   * Indicates whether some other object is "equal to" this one.
   *
   * @param obj		the reference object with which to compare.
   * @return		true if this object is the same as the obj argument;
   * 			false otherwise.
   */
  public boolean equals(Object obj) {
    if (obj instanceof ConnectionParameters)
      return (compareTo((ConnectionParameters) obj) == 0);
    else
      return false;
  }

  /**
   * Returns a shortened URL.
   *
   * @return		the shortened URL
   */
  public String toString() {
    return m_URL.replaceAll(".*\\/\\/", "");
  }

  /**
   * Returns the instance of a new database connection object.
   *
   * @param dbcon	the database connection object class to instantiate
   * @return		the new database connection object
   */
  public AbstractDatabaseConnection toDatabaseConnection(Class dbcon) {
    AbstractDatabaseConnection	result;

    try {
      result = (AbstractDatabaseConnection) dbcon.newInstance();
      result.setURL(getURL());
      result.setUser(getUser());
      result.setPassword(getPassword());
      result.setDebugLevel(getDebugLevel());
      result.setConnectOnStartUp(getConnectOnStartUp());
    }
    catch (Exception e) {
      System.err.println("Failed to create database connection object:");
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Creates a new object based on the classname, falls back to the default
   * class, if instantiation fails.
   *
   * @param classname	the class to instantiate
   * @return		the new object
   */
  public static ConnectionParameters forName(String classname) {
    ConnectionParameters	result;

    try {
      result = (ConnectionParameters) Class.forName(classname).newInstance();
    }
    catch (Exception e) {
      result = new ConnectionParameters();
    }

    return result;
  }
}