/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractControlActor.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.HashSet;

import adams.core.Pausable;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorHandler;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.ActorUtils;
import adams.flow.core.Compatibility;
import adams.flow.core.ControlActor;
import adams.flow.core.InputConsumer;

/**
 * Ancestor for all actors that control sub-actors in some way.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractControlActor
  extends AbstractActor
  implements ControlActor, ActorHandler, Pausable {

  /** for serialization. */
  private static final long serialVersionUID = -7471817724012995179L;

  /** the compatibility class in use. */
  protected Compatibility m_Compatibility;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Compatibility = new Compatibility();
  }

  /**
   * Sets whether the actor is to be run in headless mode, i.e., suppressing
   * GUI components.
   *
   * @param value	if true then GUI components will be suppressed
   */
  public void setHeadless(boolean value) {
    int		i;

    super.setHeadless(value);

    for (i = 0; i < size(); i++)
      get(i).setHeadless(value);
  }

  /**
   * Sets the parent of this actor, e.g., the group it belongs to.
   *
   * @param value	the new parent
   */
  public void setParent(AbstractActor value) {
    super.setParent(value);
    if (value != null)
      updateParent();
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public abstract ActorHandlerInfo getActorHandlerInfo();

  /**
   * Returns the size of the group.
   *
   * @return		the size
   */
  public abstract int size();

  /**
   * Returns the number of non-skipped actors.
   *
   * @return		the 'active' actors
   */
  public int active() {
    int		result;
    int		i;

    result = 0;
    for (i = 0; i < size(); i++) {
      if (!get(i).getSkip())
	result++;
    }

    return result;
  }

  /**
   * Returns the first non-skipped actor.
   *
   * @return		the first 'active' actor, null if none available
   */
  public AbstractActor firstActive() {
    AbstractActor	result;
    int			i;

    result = null;
    for (i = 0; i < size(); i++) {
      if (!get(i).getSkip()) {
	result = get(i);
	break;
      }
    }

    return result;
  }

  /**
   * Returns the last non-skipped actor.
   *
   * @return		the last 'active' actor, null if none available
   */
  public AbstractActor lastActive() {
    AbstractActor	result;
    int			i;

    result = null;
    for (i = size() - 1; i >= 0; i--) {
      if (!get(i).getSkip()) {
	result = get(i);
	break;
      }
    }

    return result;
  }

  /**
   * Returns the first non-skipped InputConsumer.
   *
   * @return		the first 'active' InputConsumer, null if none available
   */
  public AbstractActor firstInputConsumer() {
    AbstractActor	result;
    int			i;

    result = null;
    for (i = 0; i < size(); i++) {
      if (!get(i).getSkip() && (get(i) instanceof InputConsumer)) {
	result = get(i);
	break;
      }
    }

    return result;
  }

  /**
   * Returns the actor at the given position.
   *
   * @param index	the position
   * @return		the actor
   */
  public abstract AbstractActor get(int index);

  /**
   * Sets the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to set at this position
   */
  public abstract void set(int index, AbstractActor actor);

  /**
   * Returns the index of the actor.
   *
   * @param actor	the name of the actor to look for
   * @return		the index of -1 if not found
   */
  public abstract int indexOf(String actor);

  /**
   * Updates the parent of all actors in this group.
   */
  protected void updateParent() {
    int		i;

    for (i = 0; i < size(); i++) {
      get(i).setParent(null);
      get(i).setParent(this);
    }
  }

  /**
   * Checks whether the class' options can be inspected. By default, arrays
   * of actors (i.e., the control actor's sub-actors) won't be inspected, as
   * they do it themselves.
   *
   * @param cls		the class to check
   * @return		true if it can be inspected, false otherwise
   */
  protected boolean canInspectOptions(Class cls) {
    // we don't inspect sub-actors!
    if (cls == AbstractActor[].class)
      return false;
    else if (cls == AbstractActor.class)
      return false;
    else
      return super.canInspectOptions(cls);
  }

  /**
   * Performs the setUp of the sub-actors.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String setUpSubActors() {
    int			i;
    String		result;
    HashSet<String>	names;

    result = null;

    // check whether everything is correctly setup
    names = new HashSet<String>();
    for (i = 0; i < size(); i++) {
      // make sure that name is unique!
      ActorUtils.uniqueName(get(i), names);
      names.add(get(i).getName());

      // setup actor
      if (!get(i).getSkip()) {
	result = get(i).setUp();
	if (result != null) {
	  result = get(i).getFullName() + ": " + result;
	  break;
	}
      }
    }

    return result;
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;
    String		msg;

    result = super.setUp();

    if (result == null)
      result = setUpSubActors();

    if (result == null) {
      // check connections
      msg = check();
      if (isDebugOn())
	debug("check: " + ((msg == null) ? "OK" : msg));
      if (msg != null)
        result = "Check failed: " + msg;
    }

    return result;
  }

  /**
   * Performs checks on the "sub-actors". Default implementation does nothing.
   *
   * @return		null
   */
  public String check() {
    return null;
  }

  /**
   * Finishes up the execution.
   */
  public void wrapUp() {
    int		i;

    for (i = 0; i < size(); i++) {
      if (!get(i).getSkip())
	get(i).wrapUp();
    }

    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    int		i;

    for (i = 0; i < size(); i++) {
      if (!get(i).getSkip())
	get(i).cleanUp();
    }

    super.cleanUp();
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Cleans up the options and calls the destroy() method of all sub-actors.
   */
  public void destroy() {
    int		i;

    for (i = 0; i < size(); i++)
      get(i).destroy();

    super.destroy();
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    int		i;

    // pause sub-actors
    for (i = 0; i < size(); i++) {
      if (get(i) instanceof Pausable)
	((Pausable) get(i)).pauseExecution();
    }
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    return false;
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    int		i;

    // resume sub-actors
    for (i = 0; i < size(); i++) {
      if (get(i) instanceof Pausable)
	((Pausable) get(i)).resumeExecution();
    }
  }
}
