/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractControlActor.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;


/**
 * Ancestor for all actors that control sub-actors in some way.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractDirectedControlActor
  extends AbstractControlActor {

  /** for serialization. */
  private static final long serialVersionUID = -7471817724012995179L;

  /** the director used for executing. */
  protected AbstractDirector m_Director;

  /**
   * Returns an instance of a director.
   *
   * @return		the director
   */
  protected AbstractDirector newDirector() {
    return new SequentialDirector();
  }

  /**
   * Gets called when the actor needs to be re-setUp when a variable changes.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String updateVariables() {
    if (m_Director != null) {
      m_Director.stopExecution();
      m_Director.cleanUp();
      m_Director = null;
    }

    return super.updateVariables();
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;

    result = super.setUp();

    if (result == null) {
      m_Director = newDirector();
      m_Director.setControlActor(this);
      m_Director.setDebugLevel(getDebugLevel());
      m_Director.updatePrefix();
    }

    return result;
  }

  /**
   * Executes the actor.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;

    result = null;

    try {
      result = m_Director.execute();
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }

  /**
   * Finishes up the execution.
   */
  public void wrapUp() {
    // wait for director to finish up
    if (m_Director != null) {
      while (!m_Director.isFinished()) {
	synchronized(this) {
	  try {
	    wait(100);
	  }
	  catch (Exception e) {
	    // ignored
	  }
	}
      }
    }

    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    if (m_Director != null) {
      m_Director.cleanUp();
      m_Director.setControlActor(null);
      m_Director = null;
    }

    super.cleanUp();
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    super.pauseExecution();

    if (m_Director != null)
      m_Director.pauseExecution();
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    if (m_Director != null)
      return m_Director.isPaused();
    else
      return super.isPaused();
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    super.resumeExecution();

    if (m_Director != null)
      m_Director.resumeExecution();
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    if (m_Director != null)
      m_Director.stopExecution();

    super.stopExecution();
  }
}
