/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Switch.java
 * Copyright (C) 2010-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Vector;

import adams.flow.condition.bool.AbstractBooleanCondition;
import adams.flow.condition.bool.True;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.ActorUtils;
import adams.flow.core.InputConsumer;
import adams.flow.core.MutableActorHandler;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.flow.sink.Null;

/**
 <!-- globalinfo-start -->
 * Emulates a Switch control statement. The first 'condition' that evaluates to 'true' executes the corresponding 'case' actor and stops evaluation of conditions.<br/>
 * A catch-all or default can be set up as well by having one more 'case' than 'conditions' (the last case acts as default).
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Switch
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-condition &lt;adams.flow.condition.bool.AbstractBooleanCondition&gt; [-condition ...] (property: conditions)
 * &nbsp;&nbsp;&nbsp;The switch conditions to evaluate - the first condition that evaluates to 
 * &nbsp;&nbsp;&nbsp;'true' triggers the execution of the corresponding 'case' actor.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.condition.bool.True
 * </pre>
 * 
 * <pre>-case &lt;adams.flow.core.AbstractActor&gt; [-case ...] (property: cases)
 * &nbsp;&nbsp;&nbsp;The 'cases' - one of them gets executed if the corresponding 'condition' 
 * &nbsp;&nbsp;&nbsp;evaluates to 'true'.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.sink.Null
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class Switch
  extends AbstractDirectedControlActor
  implements InputConsumer, MutableActorHandler {

  /** for serialization. */
  private static final long serialVersionUID = 185561131623293880L;

  /**
   * A specialized director for the Switch control actor.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public static class SwitchDirector
    extends AbstractDirector {

    /** for serialization. */
    private static final long serialVersionUID = 8414511259688024553L;

    /** the token to use in the switch cases. */
    protected transient Token m_CaseToken;

    /**
     * Sets the group to execute.
     *
     * @param value 	the group
     */
    public void setControlActor(AbstractDirectedControlActor value) {
      if ((value instanceof Switch) || (value == null))
	super.setControlActor(value);
      else
	System.err.println(
	    "Group must be a Switch actor (provided: "
	    + ((value != null) ? value.getClass().getName() : "-null-") + ")!");
    }

    /**
     * Sets the token to use in the switch cases.
     *
     * @param value	the token to use
     */
    public void setCaseToken(Token value) {
      m_CaseToken = value;
    }

    /**
     * Returns the token to be forwarded to switch cases.
     *
     * @return		the token, can be null if not yet set
     */
    public Token getCaseToken() {
      return m_CaseToken;
    }

    /**
     * Determines which case to execute.
     *
     * @return		the index of the case to execute
     */
    protected int whichCase() {
      int	result;
      int	i;
      Switch	switchActor;

      result = -1;

      switchActor = (Switch) m_ControlActor;

      for (i = 0; i < switchActor.getConditions().length; i++) {
	try {
	  if (switchActor.getConditions()[i].evaluate(switchActor, m_CaseToken)) {
	    result = i;
	    break;
	  }
	}
	catch (Exception e) {
	  getSystemErr().println("Error evaluating boolean condition: " + switchActor.getConditions());
	  getSystemErr().printStackTrace(e);
	}
      }

      // default case?
      if (result == -1) {
	if (switchActor.getCases().length > switchActor.getConditions().length)
	  result = switchActor.getCases().length - 1;
      }

      return result;
    }

    /**
     * Executes the group of actors.
     *
     * @return		null if everything went smooth
     */
    public String execute() {
      String		result;
      String		msg;
      AbstractActor	caseActor;
      int		index;

      result = null;

      index = whichCase();
      if (index == -1) {
	result = "No matching case found!";
      }
      else {
	caseActor = ((Switch) m_ControlActor).get(index);
	try {
	  if (caseActor instanceof InputConsumer)
	    ((InputConsumer) caseActor).input(m_CaseToken);
	  result = caseActor.execute();
	}
	catch (Exception e) {
	  msg    = caseActor.getFullName() + " generated the following exception: ";
	  result = msg + e;
	  getSystemErr().println(msg);
	  getSystemErr().printStackTrace(e);
	}

	if (result != null)
	  caseActor.handleError("execute", result);
      }

      return result;
    }

    /**
     * Cleans up data structures, frees up memory.
     */
    public void cleanUp() {
      m_CaseToken = null;

      super.cleanUp();
    }
  }

  /** the key for storing the current input token in the backup. */
  public final static String BACKUP_INPUT = "input";

  /** the "conditions" for the various switch cases. */
  protected AbstractBooleanCondition[] m_Conditions;

  /** the "cases" to execute if the corresponding expression matches. */
  protected Vector<AbstractActor> m_Cases;

  /** the input token. */
  protected transient Token m_InputToken;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Emulates a Switch control statement. The first 'condition' that evaluates to 'true' "
      + "executes the corresponding 'case' actor and stops evaluation of conditions.\n"
      + "A catch-all or default can be set up as well by having one more 'case' than "
      + "'conditions' (the last case acts as default).";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "condition", "conditions",
	    new AbstractBooleanCondition[]{new True()});

    m_OptionManager.add(
	    "case", "cases",
	    new AbstractActor[]{new Null()});
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_InputToken = null;
    m_Cases      = new Vector<AbstractActor>();
  }

  /**
   * Returns an instance of a director.
   *
   * @return		the director
   */
  protected AbstractDirector newDirector() {
    return new SwitchDirector();
  }

  /**
   * Sets the conditions to evaluate.
   *
   * @param value	the conditions
   */
  public void setConditions(AbstractBooleanCondition[] value) {
    m_Conditions = value;
    reset();
  }

  /**
   * Returns the conditions to evaluate.
   *
   * @return		the conditions
   */
  public AbstractBooleanCondition[] getConditions() {
    return m_Conditions;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String conditionsTipText() {
    return
        "The switch conditions to evaluate - the first condition that "
      + "evaluates to 'true' triggers the execution of the corresponding "
      + "'case' actor.";
  }

  /**
   * Sets the cases.
   *
   * @param value 	the cases
   */
  public void setCases(AbstractActor[] value) {
    int		i;

    ActorUtils.uniqueNames(value);

    m_Cases.clear();
    for (i = 0; i < value.length; i++)
      m_Cases.add(value[i]);

    updateParent();
    reset();
  }

  /**
   * Returns the cases.
   *
   * @return 		the cases
   */
  public AbstractActor[] getCases() {
    return m_Cases.toArray(new AbstractActor[m_Cases.size()]);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String casesTipText() {
    return
        "The 'cases' - one of them gets executed if the corresponding "
      + "'condition' evaluates to 'true'.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;

    result = m_Conditions.length + " condition";
    if (m_Conditions.length != 1)
      result += "s";

    return result;
  }

  /**
   * Returns the size of the group.
   *
   * @return		the number of cases
   */
  public int size() {
    return m_Cases.size();
  }

  /**
   * Returns the actor at the given position.
   *
   * @param index	the position
   * @return		the actor
   */
  public AbstractActor get(int index) {
    return m_Cases.get(index);
  }

  /**
   * Sets the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to set at this position
   */
  public void set(int index, AbstractActor actor) {
    if ((index > -1) && (index < m_Cases.size())) {
      ActorUtils.uniqueName(actor, this, index);
      m_Cases.set(index, actor);
      reset();
      updateParent();
    }
    else {
      getSystemErr().println("Index out of range (0-" + (m_Cases.size() - 1) + "): " + index);
    }
  }

  /**
   * Inserts the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to insert
   */
  public void add(int index, AbstractActor actor) {
    m_Cases.add(index, actor);
    reset();
    updateParent();
  }

  /**
   * Removes the actor at the given position and returns the removed object.
   *
   * @param index	the position
   * @return		the removed actor
   */
  public AbstractActor remove(int index) {
    AbstractActor	result;

    result = m_Cases.remove(index);
    reset();

    return result;
  }

  /**
   * Removes all actors.
   */
  public void removeAll() {
    m_Cases.clear();
    reset();
  }

  /**
   * Returns the index of the actor.
   *
   * @param actor	the name of the actor to look for
   * @return		the index of -1 if not found
   */
  public int indexOf(String actor) {
    int		result;
    int		i;

    result = -1;

    for (i = 0; i < m_Cases.size(); i++) {
      if (m_Cases.get(i).getName().equals(actor)) {
	result = i;
	break;
      }
    }

    return result;
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(false, ActorExecution.PARALLEL, true);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    HashSet<Class>	result;

    result = new HashSet<Class>();

    if (m_Cases.size() == 0) {
      result.add(Unknown.class);
    }
    else {
      for (AbstractActor actor: m_Cases) {
	if (actor instanceof InputConsumer)
	  result.addAll(Arrays.asList(((InputConsumer) actor).accepts()));
      }
    }

    return result.toArray(new Class[result.size()]);
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_INPUT);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_InputToken != null)
      result.put(BACKUP_INPUT, m_InputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_INPUT)) {
      m_InputToken = (Token) state.get(BACKUP_INPUT);
      ((SwitchDirector) m_Director).setCaseToken(m_InputToken);
      state.remove(BACKUP_INPUT);
    }

    super.restoreState(state);
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if ((m_Conditions == null) || (m_Conditions.length == 0)) {
	result = "No condition provided!";
      }
      else if (m_Cases.size() > m_Conditions.length + 1) {
	result =   "Only 1 default case can be defined: "
	         + m_Conditions + " condition(s) but " + m_Cases.size() + " case(s)";
      }
      else if (m_Cases.size() < m_Conditions.length) {
	result = "Not enough cases defined: " + m_Conditions.length + " required";
      }
    }

    return result;
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
    m_InputToken = token;
    ((SwitchDirector) m_Director).setCaseToken(m_InputToken);
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    m_InputToken = null;

    super.cleanUp();
  }
}
