/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Trigger.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.ActorUtils;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Executes the tee-actor whenever a token gets passed through. In contrast to the Tee actor, it doesn't feed the tee-actor with the current token.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Trigger
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-progress (property: showProgress)
 * &nbsp;&nbsp;&nbsp;If set to true, progress information will be output to stdout ('.').
 * </pre>
 *
 * <pre>-stop-on-errors (property: stopOnErrors)
 * &nbsp;&nbsp;&nbsp;If set to true, errors (like exceptions) will stop the flow execution; otherwise
 * &nbsp;&nbsp;&nbsp;it is attempted to continue.
 * </pre>
 *
 * <pre>-tee &lt;adams.flow.core.AbstractActor [options]&gt; (property: teeActor)
 * &nbsp;&nbsp;&nbsp;The actor to siphon-off the tokens to.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.sink.Null
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class Trigger
  extends AbstractTee {

  /** for serialization. */
  private static final long serialVersionUID = 4690934665757923783L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Executes the tee-actor whenever a token gets passed through. In "
      + "contrast to the Tee actor, it doesn't feed the tee-actor with the "
      + "current token.";
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Actors.setAllowStandalones(true);
    m_Actors.setAllowSource(true);
  }

  /**
   * Gets called in the setUp() method. Returns null if tee-actors are fine,
   * otherwise error message.
   *
   * @return		always null
   */
  protected String setUpTeeActors() {
    return null;
  }

  /**
   * Returns whether singletons are allowed in this group or not.
   *
   * @return		true if singletons are allowed
   */
  public boolean canContainStandalones() {
    return true;
  }

  /**
   * Checks the tee actor before it is set.
   * Returns an error message if the actor is not acceptable, null otherwise.
   *
   * @param index	the index the actor gets set
   * @param actor	the actor to check
   * @return		always null
   */
  protected String checkTeeActor(int index, AbstractActor actor) {
    return null;
  }

  /**
   * Checks the tee actors before they are set via the setTeeActors method.
   * Returns an error message if the actors are not acceptable, null otherwise.
   *
   * @param actors	the actors to check
   * @return		null if checks passed or null in case of an error
   */
  protected String checkTeeActors(AbstractActor[] actors) {
    return ActorUtils.checkForSource(actors);
  }

  /**
   * Processes the token normal, i.e., not in thread.
   *
   * @param token	the token to process
   * @return		an optional error message, null if everything OK
   */
  protected String processInput(Token token) {
    String	result;

    result = m_Actors.execute();

    return result;
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(true, ActorExecution.SEQUENTIAL, false);
  }
}
