/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ActorStatistic.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.core;

import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import adams.data.statistics.InformativeStatistic;

/**
 * Generates some statistics for an actor.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ActorStatistic
  implements InformativeStatistic {

  /** the overall count. */
  public static String COUNT_ACTORS = "Actors";

  /** the control actor count. */
  public static String COUNT_CONTROLACTORS = "Control actors";

  /** the standalone count. */
  public static String COUNT_STANDALONES = "Standalones";

  /** the source count. */
  public static String COUNT_SOURCES = "Sources";

  /** the transformer count. */
  public static String COUNT_TRANSFORMERS = "Transformers";

  /** the sink count. */
  public static String COUNT_SINKS = "Sinks";

  /** the statistics. */
  protected Hashtable<String,Integer> m_Statistics;

  /** the actor to create the statistics for. */
  protected AbstractActor m_Actor;

  /**
   * Initializes the statistics.
   */
  public ActorStatistic() {
    super();

    m_Statistics = new Hashtable<String,Integer>();
    m_Actor      = null;
  }

  /**
   * Initializes the statistics with the specified actor.
   *
   * @param actor	the actor to generate the stats for
   */
  public ActorStatistic(AbstractActor actor) {
    this();
    setActor(actor);
  }

  /**
   * Sets the actor to generate the statistics for.
   *
   * @param value	the actor to use
   */
  public void setActor(AbstractActor value) {
    m_Actor = value;
    calculate();
  }

  /**
   * Returns the underlying actor.
   *
   * @return		the actor, null if none set
   */
  public AbstractActor getActor() {
    return m_Actor;
  }

  /**
   * Updates the statistics with the specified actor.
   *
   * @param actor	the actor to use
   */
  protected void update(AbstractActor actor) {
    m_Statistics.put(COUNT_ACTORS, m_Statistics.get(COUNT_ACTORS) + 1);
    if (ActorUtils.isControlActor(actor))
	m_Statistics.put(COUNT_CONTROLACTORS, m_Statistics.get(COUNT_CONTROLACTORS) + 1);
    if (ActorUtils.isStandalone(actor))
	m_Statistics.put(COUNT_STANDALONES, m_Statistics.get(COUNT_STANDALONES) + 1);
    if (ActorUtils.isSource(actor))
	m_Statistics.put(COUNT_SOURCES, m_Statistics.get(COUNT_SOURCES) + 1);
    if (ActorUtils.isTransformer(actor))
	m_Statistics.put(COUNT_TRANSFORMERS, m_Statistics.get(COUNT_TRANSFORMERS) + 1);
    if (ActorUtils.isSink(actor))
	m_Statistics.put(COUNT_SINKS, m_Statistics.get(COUNT_SINKS) + 1);
  }

  /**
   * Generates the statistics.
   */
  protected void calculate() {
    Vector<AbstractActor>	actors;

    // init stats
    m_Statistics.clear();
    m_Statistics.put(COUNT_ACTORS, 0);
    m_Statistics.put(COUNT_CONTROLACTORS, 0);
    m_Statistics.put(COUNT_STANDALONES, 0);
    m_Statistics.put(COUNT_SOURCES, 0);
    m_Statistics.put(COUNT_TRANSFORMERS, 0);
    m_Statistics.put(COUNT_SINKS, 0);

    if (m_Actor == null)
      return;

    actors = ActorUtils.enumerate(getActor());
    update(m_Actor);
    for (AbstractActor actor: actors)
      update(actor);
  }

  /**
   * Returns a description for this statistic.
   *
   * @return		the description
   */
  public String getStatisticDescription() {
    if (m_Actor == null)
      return "Statistics";
    else
      return m_Actor.getName();
  }

  /**
   * Returns all the names of the available statistical values.
   *
   * @return		the enumeration of names
   */
  public Enumeration<String> statisticNames() {
    Vector<String>	result;

    result = new Vector<String>(m_Statistics.keySet());
    Collections.sort(result);

    return result.elements();
  }

  /**
   * Returns the statistical value for the given statistic name.
   *
   * @param name	the name of the statistical value
   * @return		the corresponding value
   */
  public double getStatistic(String name) {
    return m_Statistics.get(name).doubleValue();
  }

  /**
   * Returns a string representation of the statistic.
   *
   * @return		the string representation
   */
  public String toString() {
    StringBuilder	result;
    Enumeration<String>	names;
    String		name;

    result = new StringBuilder();
    result.append(getStatisticDescription());
    result.append("\n");

    names = statisticNames();
    while (names.hasMoreElements()) {
      name = names.nextElement();
      result.append(name + ": " + getStatistic(name));
      result.append("\n");
    }

    return result.toString();
  }
}
