/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SequencePlotterPanel.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.sink.sequenceplotter;

import java.util.Vector;

import adams.gui.event.DataChangeEvent;
import adams.gui.visualization.core.AbstractColorProvider;
import adams.gui.visualization.sequence.XYSequenceContainerManager;
import adams.gui.visualization.sequence.XYSequencePanel;

/**
 * The panel that plots all the sequences.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @param <M> the container manager for markers
 */
public class SequencePlotterPanel
  extends XYSequencePanel {

  /** for serialization. */
  private static final long serialVersionUID = -325993535017871634L;

  /** the paintlet to use for marker paintlets. */
  protected MarkerPaintlet m_MarkerPaintlet;

  /** the manager for the marker sequences. */
  protected XYSequenceContainerManager m_MarkerContainerManager;

  /**
   * Initializes the panel with the specified title.
   *
   * @param title	the title to use
   */
  public SequencePlotterPanel(String title) {
    super(title);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_MarkerContainerManager = newMarkerContainerManager();
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    super.initGUI();

    m_MarkerPaintlet = new VerticalMarkers();
    m_MarkerPaintlet.setPanel(this);
    setAllowResize(true);
  }

  /**
   * Returns the manager for the marker sequences.
   *
   * @return		the new container manager
   */
  protected XYSequenceContainerManager newMarkerContainerManager() {
    XYSequenceContainerManager	result;

    result = new XYSequenceContainerManager(this);
    result.removeDataChangeListener(this);  // we don't need notifications!

    return result;
  }

  /**
   * Returns the marker container manager in use.
   *
   * @return		the container manager
   */
  public XYSequenceContainerManager getMarkerContainerManager() {
    return m_MarkerContainerManager;
  }

  /**
   * Sets the marker paintlet to use.
   *
   * @param value 	the marker paintlet
   */
  public void setMarkerPaintlet(MarkerPaintlet value) {
    m_MarkerPaintlet.setPanel(null);
    m_MarkerPaintlet = value;
    m_MarkerPaintlet.setPanel(this);
    reset();
  }

  /**
   * Returns the marker paintlet to use.
   *
   * @return 		the marker paintlet
   */
  public MarkerPaintlet getMarkerPaintlet() {
    return m_MarkerPaintlet;
  }

  /**
   * Sets the color provider to use.
   *
   * @param value	the color provider
   */
  public void setColorProvider(AbstractColorProvider value) {
    getContainerManager().setColorProvider(value.shallowCopy(true));
    getMarkerContainerManager().setColorProvider(getColorProvider().shallowCopy(true));
  }

  /**
   * Returns the color provider to use.
   *
   * @return		the color provider in use
   */
  public AbstractColorProvider getColorProvider() {
    return getContainerManager().getColorProvider();
  }

  /**
   * Translates the indices of the sequence container manager into the
   * ones from the marker container manager.
   *
   * @param seqIndices	the indices of the sequence container manager
   * @return		the indices of the marker container manager
   */
  protected int[] toMarkerIndices(int[] seqIndices) {
    int[]			result;
    Vector<Integer>		markerIndices;
    String			id;
    int				index;
    int				i;
    XYSequenceContainerManager	seqManager;
    XYSequenceContainerManager	markerManager;

    markerIndices = new Vector<Integer>();
    markerManager = getMarkerContainerManager();
    seqManager    = getContainerManager();
    for (i = 0; i < seqIndices.length; i++) {
      id    = seqManager.get(seqIndices[i]).getID();
      index = markerManager.indexOf(id);
      if (index != -1)
	markerIndices.add(index);
    }

    result = new int[markerIndices.size()];
    for (i = 0; i < markerIndices.size(); i++)
      result[i] = markerIndices.get(i);

    return result;
  }

  /**
   * Gets called if the data of the container panel has changed.
   *
   * @param e		the event that the container panel sent
   */
  public void dataChanged(DataChangeEvent e) {
    int[]		indices;
    int			i;

    switch (e.getType()) {
      case CLEAR:
	getMarkerContainerManager().clear();
	break;

      case REMOVAL:
	indices = toMarkerIndices(e.getIndices());
	for (i = indices.length - 1; i >= 0; i--)
	  getMarkerContainerManager().remove(indices[i]);
	break;

      case VISIBILITY:
	indices = toMarkerIndices(e.getIndices());
	for (i = 0; i < indices.length; i++)
	  getMarkerContainerManager().setVisible(indices[i], getContainerManager().isVisible(indices[i]));
	break;

      case BULK_UPDATE:
	// let's remove non-existent sequences
	i = 0;
	while (i < getMarkerContainerManager().count()) {
	  if (getContainerManager().indexOf(getMarkerContainerManager().get(i).getID()) == -1)
	    getMarkerContainerManager().remove(i);
	  else
	    i++;
	}
	break;
    }

    super.dataChanged(e);
  }
}
