/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Tree.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.debug.objecttree;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import javax.swing.tree.DefaultTreeModel;

import adams.core.option.AbstractArgumentOption;
import adams.core.option.AbstractOption;
import adams.core.option.OptionHandler;
import adams.gui.core.BaseTree;
import adams.gui.visualization.debug.inspectionhandler.AbstractInspectionHandler;

/**
 * Specialized tree that displays the properties of an object.
 * <p/>
 * In order to avoid loops, a HashSet is used for keeping track of the processed
 * objects. Of course, custom equals(Object)/compareTo(Object) methods will
 * interfere with this mechanism.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class Tree
  extends BaseTree {

  /** for serialization. */
  private static final long serialVersionUID = -127345486742553561L;

  /** the tree itself. */
  protected Tree m_Self;

  /** the current object. */
  protected transient Object m_Object;

  /** the inspection handlers. */
  protected Vector<AbstractInspectionHandler> m_Handlers;

  /** the objects that have been inspected already. */
  protected HashSet<Object> m_Inspected;

  /**
   * Initializes the tree.
   */
  public Tree() {
    super();

    m_Self = this;
    setShowsRootHandles(true);
    setRootVisible(true);
    setCellRenderer(new Renderer());
    initHandlers();
    buildTree(null);
  }

  /**
   * Initializes the handlers.
   */
  protected void initHandlers() {
    String[]	handlers;

    handlers   = AbstractInspectionHandler.getHandlers();
    m_Handlers = new Vector<AbstractInspectionHandler>();
    for (String handler: handlers) {
      try {
	m_Handlers.add((AbstractInspectionHandler) Class.forName(handler).newInstance());
      }
      catch (Exception e) {
	System.err.println("Failed to instantiate inspection handler '" + handler + "':");
	e.printStackTrace();
      }
    }
  }

  /**
   * Builds the tree from the given object.
   *
   * @param root	the object to build the tree from, null for empty tree
   */
  protected void buildTree(Object root) {
    DefaultTreeModel		model;
    Node		rootNode;

    m_Inspected = new HashSet<Object>();

    if (root == null) {
      model = new DefaultTreeModel(null);
    }
    else {
      rootNode = buildTree(null, null, root, false);
      model    = new DefaultTreeModel(rootNode);
    }

    m_Inspected.clear();

    setModel(model);
  }

  /**
   * Adds the array below the parent.
   *
   * @param parent	the parent to add the array to
   * @param obj		the array to add
   */
  protected void addArray(Node parent, Object obj) {
    int		len;
    int		i;
    Object	value;

    len = Array.getLength(obj);
    for (i = 0; i < len; i++) {
      value = Array.get(obj, i);
      buildTree(parent, "[" + (i+1) + "]", value, true);
    }
  }

  /**
   * Builds the tree recursively.
   *
   * @param parent	the parent to add the object to (null == root)
   * @param property	the name of the property the object belongs to (null == root)
   * @param obj		the object to add
   * @param element	whether the object is element of an array
   * @return		the generated node
   */
  protected Node buildTree(Node parent, String property, Object obj, boolean element) {
    Node			result;
    BeanInfo 			bi;
    PropertyDescriptor[]	descriptors;
    Object			current;
    List<AbstractOption>	options;
    Hashtable<String,Object>	additional;
    String			label;
    int				i;

    result = new Node(property, obj, element);
    if (parent != null)
      parent.add(result);

    m_Inspected.add(obj);

    if (obj.getClass().isArray())
      addArray(result, obj);

    // child properties
    try {
      if (obj instanceof OptionHandler) {
	options = ((OptionHandler) obj).getOptionManager().getOptionsList();
	for (AbstractOption option: options) {
	  current = option.getCurrentValue();
	  if (current == null)
	    continue;
	  if (m_Inspected.contains(current))
	    continue;
	  label = option.getProperty();
	  if (option instanceof AbstractArgumentOption) {
	    if (((AbstractArgumentOption) option).isVariableAttached())
	      label += "/" + ((AbstractArgumentOption) option).getVariable();
	  }
	  buildTree(result, label, current, false);
	}
      }
      else {
	bi          = Introspector.getBeanInfo(obj.getClass());
	descriptors = bi.getPropertyDescriptors();
	for (PropertyDescriptor desc: descriptors) {
	  try {
	    if ((desc.getReadMethod() != null) && (desc.getWriteMethod() != null)) {
	      current = desc.getReadMethod().invoke(obj, new Object[0]);
	      if (current == null)
		continue;
	      if (m_Inspected.contains(current))
		continue;
	      if (current.getClass() != Class.class)
		buildTree(result, desc.getDisplayName(), current, false);
	    }
	  }
	  catch (Exception e) {
	    System.err.println("Failed to obtain current value for: " + desc.getDisplayName());
	    e.printStackTrace();
	  }
	}
      }
    }
    catch (Exception e) {
      System.err.println("Failed to obtain property descriptors for: " + obj);
      e.printStackTrace();
    }

    // additional values obtained through inspection handlers
    for (i = 0; i < m_Handlers.size(); i++) {
      if (m_Handlers.get(i).handles(obj)) {
	additional = m_Handlers.get(i).inspect(obj);
	for (String key: additional.keySet()) {
	  buildTree(result, key, additional.get(key), false);
	}
      }
    }

    return result;
  }

  /**
   * Sets the object to display.
   *
   * @param value	the object
   */
  public void setObject(Object value) {
    m_Object = value;
    buildTree(m_Object);
  }

  /**
   * Returns the object currently displayed.
   *
   * @return		the object
   */
  public Object getObject() {
    return m_Object;
  }
}