/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * UtilsTest.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.core;

import junit.framework.Test;
import junit.framework.TestSuite;
import adams.core.Utils;
import adams.env.Environment;
import adams.test.AdamsTestCase;

/**
 * Tests the adams.core.Utils class. Run from commandline with: <p/>
 * java adams.core.UtilsTest
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class UtilsTest
  extends AdamsTestCase {

  /**
   * Initializes the test.
   *
   * @param name	the name of the test
   */
  public UtilsTest(String name) {
    super(name);
  }

  /**
   * Test the padLeft method with no truncating.
   */
  public void testPadLeftNoTruncate() {
    String 	in;
    String	out;

    in  = "blah";
    out = Utils.padLeft(in, '0', 10);
    assertEquals("Length wrong", 10, out.length());
    assertEquals("Content wrong", "000000blah", out);

    in  = "blah";
    out = Utils.padLeft(in, '0', 4);
    assertEquals("Length wrong", 4, out.length());
    assertEquals("Content wrong", in, out);

    in  = "blah";
    out = Utils.padLeft(in, '0', 3);
    assertEquals("Length wrong", 4, out.length());
    assertEquals("Content wrong", in, out);
  }

  /**
   * Test the padLeft method with truncating.
   */
  public void testPadLeftWithTruncate() {
    String 	in;
    String	out;

    in  = "blah";
    out = Utils.padLeft(in, '0', 10, true);
    assertEquals("Length wrong", 10, out.length());
    assertEquals("Content wrong", "000000blah", out);

    in  = "blah";
    out = Utils.padLeft(in, '0', 4, true);
    assertEquals("Length wrong", 4, out.length());
    assertEquals("Content wrong", in, out);

    in  = "blah";
    out = Utils.padLeft(in, '0', 3, true);
    assertEquals("Length wrong", 3, out.length());
    assertEquals("Content wrong", "lah", out);
  }

  /**
   * Performs a test for the split(String,char) method.
   *
   * @param in		the string to split
   * @param delim	the delimiter to use
   * @param exp		the expected output
   */
  protected void performSplitTest(String in, char delim, String[] exp) {
    String[]	out;
    int		i;

    out = Utils.split(in, delim);
    assertEquals("# of cells differs", exp.length, out.length);
    for (i = 0; i < exp.length; i++)
      assertEquals("cell #" + (i+1) + " differs", exp[i], out[i]);
  }

  /**
   * Tests the split(String,char) method.
   */
  public void testSplit() {
    performSplitTest("hello", ',', new String[]{"hello"});
    performSplitTest("hello,world", ',', new String[]{"hello", "world"});
    performSplitTest("hello", '|', new String[]{"hello"});
    performSplitTest("hello|world", '|', new String[]{"hello", "world"});
    performSplitTest("hello,world,out,there", ',', new String[]{"hello", "world", "out", "there"});
    performSplitTest("hello,world,out,there", '\t', new String[]{"hello,world,out,there"});
    performSplitTest("hello\tworld\tout\tthere", '\t', new String[]{"hello", "world", "out", "there"});
    performSplitTest("hello\t\tout\tthere", '\t', new String[]{"hello", "", "out", "there"});
    performSplitTest("hello\t\t\t", '\t', new String[]{"hello", "", "", ""});
  }

  /**
   * Tests the doubleToStringFixed method.
   */
  public void testDoubleToStringFixed() {
    assertEquals("output differs", "1.234", Utils.doubleToStringFixed(1.234, -1));
    assertEquals("output differs", "1", Utils.doubleToStringFixed(1.234, 0));
    assertEquals("output differs", "1.2", Utils.doubleToStringFixed(1.234, 1));
    assertEquals("output differs", "1.23", Utils.doubleToStringFixed(1.234, 2));

    assertEquals("output differs", "9.876", Utils.doubleToStringFixed(9.876, -1));
    assertEquals("output differs", "9", Utils.doubleToStringFixed(9.876, 0));
    assertEquals("output differs", "9.8", Utils.doubleToStringFixed(9.876, 1));
    assertEquals("output differs", "9.87", Utils.doubleToStringFixed(9.876, 2));
  }

  /**
   * Returns a test suite.
   *
   * @return		the test suite
   */
  public static Test suite() {
    return new TestSuite(UtilsTest.class);
  }

  /**
   * Runs the test from commandline.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    Environment.setEnvironmentClass(Environment.class);
    runTest(suite());
  }
}
