/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractFormattedSpreadSheetWriter.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.text.DecimalFormat;

/**
 * Ancestor for writers that format the numbers.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractFormattedSpreadSheetWriter
  extends AbstractSpreadSheetWriter {

  /** for serialization. */
  private static final long serialVersionUID = -9100041207747021318L;

  /** The format for the numbers. */
  protected String m_NumberFormat;

  /** the formatter to use. */
  protected transient DecimalFormat m_Formatter;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "number-format", "numberFormat",
	    getDefaultNumberFormat());
  }

  /**
   * Returns the default number format.
   *
   * @return		the default format
   */
  protected String getDefaultNumberFormat() {
    return "";
  }

  /**
   * Sets the number format.
   *
   * @param value	the format
   */
  public void setNumberFormat(String value) {
    m_NumberFormat = value;
    reset();
  }

  /**
   * Returns the number format.
   *
   * @return		the format
   */
  public String getNumberFormat() {
    return m_NumberFormat;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String numberFormatTipText() {
    return "The format for the numbers (see java.text.DecimalFormat), use empty string for default 'double' output.";
  }

  /**
   * Formats the number according to the format and returns the generated
   * textual representation.
   *
   * @param value	the double value to turn into a string
   * @return		the generated string
   */
  protected synchronized String format(double value) {
    String	result;

    if (m_NumberFormat.length() > 0) {
      if (m_Formatter == null) {
	try {
	  m_Formatter = new DecimalFormat(m_NumberFormat);
	}
	catch (Exception e) {
	  getSystemErr().println("Failed to initialize formatter with format '" + m_NumberFormat + "':");
	  getSystemErr().printStackTrace(e);
	  m_Formatter = new DecimalFormat();
	}
      }

      result = m_Formatter.format(value);
    }
    else {
      result = "" + value;
    }

    return result;
  }
}
