/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheet.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.core.io;

import java.io.Serializable;
import java.io.StringWriter;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import adams.core.CloneHandler;
import adams.core.Utils;

/**
 * Represents a spreadsheet that can be saved to CSV.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class SpreadSheet
  implements Serializable, CloneHandler<SpreadSheet> {

  /** for serialization. */
  private static final long serialVersionUID = -5500678849412481001L;

  /**
   * Represents a single cell.
   */
  public static class Cell
    implements Serializable, CloneHandler<Cell> {

    /** for serialization. */
    private static final long serialVersionUID = -3912508808391288142L;

    /** the content of the cell. */
    protected String m_Content;

    /** whether the content is numeric. */
    protected boolean m_IsNumeric;

    /**
     * default constructor.
     */
    public Cell() {
      super();

      m_Content   = "";
      m_IsNumeric = false;
    }

    /**
     * Returns a clone of itself.
     *
     * @return		the clone
     */
    public Cell getClone() {
      Cell	result;

      result             = new Cell();
      result.m_Content   = m_Content;
      result.m_IsNumeric = m_IsNumeric;

      return result;
    }

    /**
     * Sets the content of the cell.
     *
     * @param value	the content; null intepreted as missing value
     */
    public void setContent(Integer value) {
      if (value == null)
	setContent(MISSING_VALUE, true);
      else
	setContent("" + value, true);
    }

    /**
     * Sets the content of the cell.
     *
     * @param value	the content; null or NaN is intepreted as missing value
     */
    public void setContent(Double value) {
      if ((value == null) || (Double.isNaN(value)))
	setContent(MISSING_VALUE, true);
      else
	setContent("" + value, true);
    }

    /**
     * Sets the content of the cell. Tries to determine whether the cell
     * content is numeric or not.
     *
     * @param value	the content
     */
    public void setContent(String value) {
      if ((value.length() > 0) && (!value.equals(MISSING_VALUE))) {
	if (Utils.isDouble(value))
	  setContent(value, true);
	else
	  setContent(value, false);
      }
    }

    /**
     * Sets the content of the cell.
     *
     * @param value	the content
     * @param numeric	whether the content is numeric or not
     */
    public void setContent(String value, boolean numeric) {
      m_Content   = value;
      m_IsNumeric = numeric;
    }

    /**
     * Returns the content of the cell.
     *
     * @return		the content
     */
    public String getContent() {
      return m_Content;
    }

    /**
     * Checks whether the stored string is numeric.
     *
     * @return		true if the content is numeric
     */
    public boolean isNumeric() {
      return m_IsNumeric;
    }

    /**
     * Checks whether the cell contains a missing value.
     *
     * @return		true if missing value
     */
    public boolean isMissing() {
      return m_Content.equals(MISSING_VALUE);
    }

    /**
     * Returns the content of the cell.
     *
     * @return		the content
     */
    public String toString() {
      return getContent();
    }

    /**
     * Returns whether the content represents a double number.
     *
     * @return		true if a double
     */
    public boolean isDouble() {
      if (m_IsNumeric)
	return Utils.isDouble(m_Content);
      else
	return false;
    }

    /**
     * Returns the content as double, if possible.
     *
     * @return		the content as double, if representing a number,
     * 			otherwise null
     */
    public Double toDouble() {
      if (m_IsNumeric)
	return new Double(getContent());
      else
	return null;
    }

    /**
     * Returns whether the content represents a long number.
     *
     * @return		true if a long
     */
    public boolean isLong() {
      if (m_IsNumeric)
	return Utils.isLong(m_Content);
      else
	return false;
    }

    /**
     * Returns the content as long, if possible. First, a Double object is
     * created and then the longValue() method called to return the
     *
     * @return		the content as long, if representing a number,
     * 			otherwise null
     */
    public Long toLong() {
      if (m_IsNumeric)
	return new Double(getContent()).longValue();
      else
	return null;
    }
  }

  /**
   * Represents a single row.
   */
  public static class Row
    implements Serializable, CloneHandler<Row> {

    /** for serialization. */
    private static final long serialVersionUID = -5657793858002845967L;

    /** the cell keys of the row. */
    protected Vector<String> m_CellKeys;

    /** the cells of the row. */
    protected Hashtable<String, Cell> m_Cells;

    /**
     * default constructor.
     */
    public Row() {
      super();

      m_CellKeys = new Vector<String>();
      m_Cells    = new Hashtable<String, Cell>();
    }

    /**
     * Returns a clone of itself.
     *
     * @return		the clone
     */
    public Row getClone() {
      Row	result;
      int	i;

      result            = new Row();
      result.m_CellKeys = (Vector<String>) m_CellKeys.clone();
      for (i = 0; i < m_CellKeys.size(); i++)
	result.m_Cells.put(m_CellKeys.get(i), (Cell) m_Cells.get(m_CellKeys.get(i)).getClone());

      return result;
    }

    /**
     * Returns whether the row alread contains the cell with the given key.
     *
     * @param cellKey	the key to look for
     * @return		true if the cell already exists
     */
    public boolean hasCell(String cellKey) {
      return m_Cells.containsKey(cellKey);
    }

    /**
     * Adds a cell with the given key to the list and returns the created object.
     * If the cell already exists, then this cell is returned instead and no new
     * object created.
     *
     * @param cellKey	the key for the cell to create
     * @return		the created cell or the already existing cell
     */
    public Cell addCell(String cellKey) {
      Cell	result;

      if (hasCell(cellKey)) {
        result = getCell(cellKey);
      }
      else {
        result = new Cell();
        m_CellKeys.add(cellKey);
        m_Cells.put(cellKey, result);
      }

      return result;
    }

    /**
     * Returns the cell with the given key, null if not found.
     *
     * @param cellKey	the cell to look for
     * @return		the cell or null if not found
     */
    public Cell getCell(String cellKey) {
      return m_Cells.get(cellKey);
    }

    /**
     * Returns the cell with the given index, null if not found.
     *
     * @param columnIndex	the index of the column
     * @return			the cell or null if not found
     */
    public Cell getCell(int columnIndex) {
      Cell	result;
      String	key;

      result = null;
      key    = getCellKey(columnIndex);
      if (key != null)
	result = getCell(key);

      return result;
    }

    /**
     * Returns the cell content with the given index.
     *
     * @param columnIndex	the index of the column
     * @return			the content or null if not found
     */
    public String getContent(int columnIndex) {
      String	result;
      String	key;

      result = null;
      key    = getCellKey(columnIndex);
      if (key != null)
	result = getCell(key).getContent();

      return result;
    }

    /**
     * Returns the cell content with the given index as double.
     *
     * @param columnIndex	the index of the column
     * @return			the content as double or null if not found or
     * 				not numeric
     */
    public Double toDouble(int columnIndex) {
      Double	result;
      String	key;

      result = null;
      key    = getCellKey(columnIndex);
      if (key != null)
	result = getCell(key).toDouble();

      return result;
    }

    /**
     * Returns the cell key with the given column index.
     *
     * @param columnIndex	the index of the column
     * @return			the cell key, null if invalid index
     */
    public String getCellKey(int columnIndex) {
      if (columnIndex < m_CellKeys.size())
	return m_CellKeys.get(columnIndex);
      else
	return null;
    }

    /**
     * Returns an enumeration over all stored cell keys.
     *
     * @return		all cell keys
     */
    public Enumeration<String> cellKeys() {
      return m_CellKeys.elements();
    }

    /**
     * Returns the number of cells stored in the row.
     *
     * @return		the number of cells
     */
    public int getCellCount() {
      return m_Cells.size();
    }

    /**
     * Simply returns the internal hashtable of cells as string.
     *
     * @return		the values of the row
     */
    public String toString() {
      return m_Cells.toString();
    }
  }

  /** the line comment start. */
  public final static String COMMENT = "#";

  /** the default missing value. */
  public final static String MISSING_VALUE = "?";

  /** the row keys of the spreadsheet. */
  protected Vector<String> m_RowKeys;

  /** the rows of the spreadsheet. */
  protected Hashtable<String, Row> m_Rows;

  /** the header row. */
  protected Row m_HeaderRow;

  /** optional comments. */
  protected Vector<String> m_Comments;

  /** the name of the spreadsheet. */
  protected String m_Name;

  /**
   * default constructor.
   */
  public SpreadSheet() {
    super();

    m_RowKeys   = new Vector<String>();
    m_Rows      = new Hashtable<String, Row>();
    m_HeaderRow = new Row();
    m_Comments  = new Vector<String>();
    m_Name      = null;
  }

  /**
   * Returns a clone of itself.
   *
   * @return		the clone
   */
  public SpreadSheet getClone() {
    SpreadSheet	result;
    int		i;

    result             = new SpreadSheet();
    result.m_HeaderRow = (Row) m_HeaderRow.getClone();
    result.m_Comments  = (Vector<String>) m_Comments.clone();
    result.m_RowKeys   = (Vector<String>) m_RowKeys.clone();
    for (i = 0; i < m_RowKeys.size(); i++)
      result.m_Rows.put(m_RowKeys.get(i), (Row) m_Rows.get(m_RowKeys.get(i)).getClone());

    return result;
  }

  /**
   * Sets the name of the spreadsheet.
   *
   * @param value	the name
   */
  public void setName(String value) {
    m_Name = value;
  }

  /**
   * Returns the name of the spreadsheet.
   *
   * @return		the name, can be null
   */
  public String getName() {
    return m_Name;
  }

  /**
   * Returns whether the spreadsheet has a name.
   *
   * @return		true if the spreadsheet is named
   */
  public boolean hasName() {
    return (m_Name != null);
  }

  /**
   * Adds the comment to the internal list of comments.
   *
   * @param comment	the comment to add
   */
  public void addComment(String comment) {
    m_Comments.add(comment);
  }

  /**
   * Returns the comments.
   *
   * @return		the comments
   */
  public Vector<String> getComments() {
    return m_Comments;
  }

  /**
   * Returns the header row.
   *
   * @return		the row
   */
  public Row getHeaderRow() {
    return m_HeaderRow;
  }

  /**
   * Returns whether the spreadsheet already contains the row with the given key.
   *
   * @param rowKey	the key to look for
   * @return		true if the row already exists
   */
  public boolean hasRow(String rowKey) {
    return m_Rows.containsKey(rowKey);
  }

  /**
   * Adds a row with the given key to the list and returns the created object.
   * If the row already exists, then this row is returned instead and no new
   * object created.
   *
   * @param rowKey	the key for the row to create
   * @return		the created row or the already existing row
   */
  public Row addRow(String rowKey) {
    Row		result;

    if (hasRow(rowKey)) {
      result = getRow(rowKey);
    }
    else {
      result = new Row();
      m_RowKeys.add(rowKey);
      m_Rows.put(rowKey, result);
    }

    return result;
  }

  /**
   * Returns the row associated with the given row key, null if not found.
   *
   * @param rowKey	the key of the row to retrieve
   * @return		the row or null if not found
   */
  public Row getRow(String rowKey) {
    return m_Rows.get(rowKey);
  }

  /**
   * Returns the row at the specified index.
   *
   * @param rowIndex	the 0-based index of the row to retrieve
   * @return		the row
   */
  public Row getRow(int rowIndex) {
    return m_Rows.get(m_RowKeys.get(rowIndex));
  }

  /**
   * Returns the row key at the specified index.
   *
   * @param rowIndex	the 0-based index of the row key to retrieve
   * @return		the row key
   */
  public String getRowKey(int rowIndex) {
    return m_RowKeys.get(rowIndex);
  }

  /**
   * Returns the row index of the specified row.
   *
   * @param rowKey	the row identifier
   * @return		the 0-based row index, -1 if not found
   */
  public int getRowIndex(String rowKey) {
    int		result;
    int		i;

    result = -1;

    for (i = 0; i < m_RowKeys.size(); i++) {
      if (m_RowKeys.get(i).equals(rowKey)) {
	result = i;
	break;
      }
    }

    return result;
  }

  /**
   * Returns the cell index of the specified cell (in the header row).
   *
   * @param cellKey	the cell identifier
   * @return		the 0-based column index, -1 if not found
   */
  public int getCellIndex(String cellKey) {
    int		result;
    int		i;

    result = -1;

    for (i = 0; i < m_HeaderRow.getCellCount(); i++) {
      if (m_HeaderRow.getCellKey(i).equals(cellKey)) {
	result = i;
	break;
      }
    }

    return result;
  }

  /**
   * Checks whether the cell with the given keys already exists.
   *
   * @param rowKey	the key of the row to look for
   * @param cellKey	the key of the cell in the row to look for
   * @return		true if the cell exists
   */
  public boolean hasCell(String rowKey, String cellKey) {
    boolean	result;
    Row		row;

    result = hasRow(rowKey);

    if (result) {
      row    = getRow(rowKey);
      result = row.hasCell(cellKey);
    }

    return result;
  }

  /**
   * Returns the corresponding cell or null if not found.
   *
   * @param rowKey	the key of the row the cell is in
   * @param cellKey	the key of the cell to retrieve
   * @return		the cell or null if not found
   */
  public Cell getCell(String rowKey, String cellKey) {
    Cell	result;
    Row		row;

    result = null;

    row    = getRow(rowKey);
    if (row != null)
      result = row.getCell(cellKey);

    return result;
  }

  /**
   * Returns the position of the cell or null if not found. A position is a
   * combination of a number of letters (for the column) and number (for the
   * row).
   *
   * @param rowKey	the key of the row the cell is in
   * @param cellKey	the key of the cell to retrieve
   * @return		the position string or null if not found
   */
  public String getCellPosition(String rowKey, String cellKey) {
    int		rowIndex;
    int		cellIndex;

    rowIndex  = getRowIndex(rowKey);
    cellIndex = getCellIndex(cellKey);

    return getCellPosition(rowIndex + 1, cellIndex);
  }

  /**
   * Returns the position of the cell. A position is a combination of a number
   * of letters (for the column) and number (for the row).
   * <p/>
   * Note: add "1" to the row indices, since the header row does not count
   * towards the row count.
   *
   * @param row		the row index of the cell
   * @param col		the column index of the cell
   * @return		the position string or null if not found
   */
  public static String getCellPosition(int row, int col) {
    String		result;
    Vector<Integer>	digits;
    int			i;

    result = null;

    // A-Z, AA-ZZ, AAA-ZZZ
    if (col >= 26 + 676 + 17576)
      throw new IllegalArgumentException("Column of cell too large: " + col + " >= " + (26 + 676 + 17576));

    if ((row == -1) || (col == -1))
      return result;

    result = "";

    // A-Z
    if (col < 26) {
      digits = Utils.toBase(col, 26);
    }
    // AA-ZZ
    else if (col < 26 + 676) {
      digits = Utils.toBase(col - 26, 26);
      while (digits.size() < 2)
	digits.add(0);
    }
    // AAA-ZZZ
    else {
      digits = Utils.toBase(col - 26 - 676, 26);
      while (digits.size() < 3)
	digits.add(0);
    }

    for (i = digits.size() - 1; i >= 0; i--)
      result += (char) ((int) 'A' + digits.get(i));

    result += (row + 1);

    return result;
  }

  /**
   * Returns an enumeration over all row keys.
   *
   * @return		the row keys
   */
  public Enumeration<String> rowKeys() {
    return m_RowKeys.elements();
  }

  /**
   * Sorts the rows according to the row keys.
   *
   * @see	#rowKeys()
   */
  public void sort() {
    Collections.sort(m_RowKeys);
  }

  /**
   * Sorts the rows according to the row keys.
   *
   * @param comp	the comparator to use
   * @see		#rowKeys()
   */
  public void sort(Comparator<String> comp) {
    Collections.sort(m_RowKeys, comp);
  }

  /**
   * Returns the number of columns.
   *
   * @return		the number of columns
   */
  public int getColumnCount() {
    return getHeaderRow().getCellCount();
  }

  /**
   * Returns the number of rows currently stored.
   *
   * @return		the number of rows
   */
  public int getRowCount() {
    return m_RowKeys.size();
  }

  /**
   * Checks whether the given column is numeric or not.
   *
   * @param columnIndex	the index of the column to check
   * @return		true if purely numeric
   */
  public boolean isNumeric(int columnIndex) {
    boolean	result;
    int		i;
    String	colKey;
    Cell	cell;

    result = true;
    colKey = m_HeaderRow.getCellKey(columnIndex);

    for (i = 0; i < getRowCount(); i++) {
      cell = getRow(i).getCell(colKey);
      if ((cell != null) && !cell.isNumeric()) {
	result = false;
	break;
      }
    }

    return result;
  }

  /**
   * Returns the spreadsheet as string, i.e., CSV formatted.
   *
   * @return		the string representation
   */
  public String toString() {
    StringWriter	writer;

    writer = new StringWriter();
    new CsvSpreadSheetWriter().write(this, writer);

    return writer.toString();
  }
}
