/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ProcessUtils.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.core.management;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.lang.management.ManagementFactory;

/**
 * A helper class for process related stuff.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ProcessUtils {

  /** the constant for the auto PID. */
  public final static long AUTO_PID = -999;

  /** the constant for no PID. */
  public final static long NO_PID = -1;

  /**
   * Returns the PID of the virtual machine. Caution: it's a hack and can
   * break anytime. Do NOT rely on it.
   * Based on <a href="http://blog.igorminar.com/2007/03/how-java-application-can-discover-its.html" target="_blank">this blog entry</a>.
   *
   * @return		the PID or -1 in case of an error
   */
  public static long getVirtualMachinePID() {
    long	result;
    String	name;

    name = ManagementFactory.getRuntimeMXBean().getName();

    try {
      result = Long.parseLong(name.replaceAll("@.*", ""));
    }
    catch (Exception e) {
      result = -1;
    }

    return result;
  }

  /**
   * Executes the commandline array and returns the outut of stdout or stderr if
   * successful. If the command fails, it returns the error code.
   *
   * @param cmd		the commandline array to execute
   * @param stderr	if true then stderr output is returned instead of stdout
   * @return		an Integer object in case the process generated an exit
   * 			code other than 0, a String object with the generated
   * 			output otherwise
   * @throws Exception	if something goes wrong
   */
  protected static Object execute(Process process, boolean stderr) throws Exception {
    Object		result;
    int			exitVal;
    String		line;
    StringBuffer	output;
    BufferedReader	reader;

    output  = new StringBuffer();

    if (stderr)
      reader = new BufferedReader(new InputStreamReader(process.getErrorStream()), 1024);
    else
      reader = new BufferedReader(new InputStreamReader(process.getInputStream()), 1024);
    while ((line = reader.readLine()) != null) {
      output.append(line);
      output.append("\n");
    }

    exitVal = process.waitFor();

    if (exitVal != 0)
      result = new Integer(exitVal);
    else
      result = output.toString();

    return result;
  }

  /**
   * Executes the command and returns the outut of stdout or stderr if
   * successful. If the command fails, it returns the error code.
   *
   * @param cmd		the command to execute
   * @param stderr	if true then stderr output is returned instead of stdout
   * @return		an Integer object in case the process generated an exit
   * 			code other than 0, a String object with the generated
   * 			output otherwise
   * @throws Exception	if something goes wrong
   */
  public static Object execute(String cmd, boolean stderr) throws Exception {
    Runtime		runtime;
    Process		process;

    runtime = Runtime.getRuntime();
    process = runtime.exec(cmd);

    return execute(process, stderr);
  }

  /**
   * Executes the commandline array and returns the outut of stdout or stderr if
   * successful. If the command fails, it returns the error code.
   *
   * @param cmd		the commandline array to execute
   * @param stderr	if true then stderr output is returned instead of stdout
   * @return		an Integer object in case the process generated an exit
   * 			code other than 0, a String object with the generated
   * 			output otherwise
   * @throws Exception	if something goes wrong
   */
  public static Object execute(String[] cmd, boolean stderr) throws Exception {
    Runtime		runtime;
    Process		process;

    runtime = Runtime.getRuntime();
    process = runtime.exec(cmd);

    return execute(process, stderr);
  }

  /**
   * Returns the number of available processors.
   *
   * @return		the number of processors of the machine
   */
  public static int getAvailableProcessors() {
    return Runtime.getRuntime().availableProcessors();
  }
}
