/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * MultiArrayStatistic.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.data.statistics;

/**
 <!-- globalinfo-start -->
 * Merges the spreadsheets generated by its sub-statistics into a single one.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-statistic &lt;adams.data.statistics.AbstractArrayStatistic [options]&gt; [-statistic ...] (property: subStatistics)
 * &nbsp;&nbsp;&nbsp;The base statistics to accumulate in a single spreadsheet (side-by-side).
 * &nbsp;&nbsp;&nbsp;default: adams.data.statistics.ArrayMedian
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @param <T> the data to process
 */
public class MultiArrayStatistic<T extends Number>
  extends AbstractArrayStatistic<T> {

  /** for serialization. */
  private static final long serialVersionUID = 6146460163351963643L;

  /** the statistics to perform and merge into single spreadsheet. */
  protected AbstractArrayStatistic[] m_SubStatistics;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Merges the spreadsheets generated by its sub-statistics into a single one.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "statistic", "subStatistics",
	    new AbstractArrayStatistic[]{new ArrayMedian()});
  }

  /**
   * Sets the base statistics to use.
   *
   * @param value 	the statistics
   */
  public void setSubStatistics(AbstractArrayStatistic[] value) {
    m_SubStatistics = value;
    reset();
  }

  /**
   * Returns the base statistics to use.
   *
   * @return 		the statistics
   */
  public AbstractArrayStatistic[] getSubStatistics() {
    return m_SubStatistics;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String subStatisticsTipText() {
    return "The base statistics to accumulate in a single spreadsheet (side-by-side).";
  }

  /**
   * Returns the minimum number of arrays that need to be present.
   * -1 for unbounded.
   *
   * @return		the minimum number, -1 for unbounded
   */
  public int getMin() {
    int		result;
    int		i;

    result = -1;

    for (i = 0; i < m_SubStatistics.length; i++) {
      if (m_SubStatistics[i].getMin() > result)
	result = m_SubStatistics[i].getMin();
    }

    return result;
  }

  /**
   * Returns the maximum number of arrays that need to be present.
   * -1 for unbounded.
   *
   * @return		the maximum number, -1 for unbounded
   */
  public int getMax() {
    int		result;
    int		i;

    result = Integer.MAX_VALUE;

    for (i = 0; i < m_SubStatistics.length; i++) {
      if ((m_SubStatistics[i].getMax() != -1) && (m_SubStatistics[i].getMax() < result))
	result = m_SubStatistics[i].getMax();
    }

    if (result == Integer.MAX_VALUE)
      result = -1;

    return result;
  }

  /**
   * Generates the actual result.
   *
   * @return		the generated result
   */
  protected StatisticContainer doCalculate() {
    StatisticContainer		result;
    StatisticContainer[]	stats;
    AbstractArrayStatistic	statistic;
    int				i;
    int				n;
    int				m;
    int				colOffset;
    int				rows;
    int				cols;

    // gather the statistics
    stats = new StatisticContainer[m_SubStatistics.length];
    for (i = 0; i < m_SubStatistics.length; i++) {
      statistic = m_SubStatistics[i].shallowCopy(true);
      for (n = 0; n < size(); n++)
	statistic.add(get(n));
      stats[i] = statistic.calculate();
    }

    // create combined statistic
    rows = 0;
    cols = 0;
    for (i = 0; i < stats.length; i++) {
      cols += stats[i].getColumnCount();
      if (stats[i].getRowCount() > rows)
	rows = stats[i].getRowCount();
    }

    result    = new StatisticContainer(rows, cols);
    colOffset = 0;
    for (i = 0; i < stats.length; i++) {
      for (n = 0; n < stats[i].getColumnCount(); n++) {
	// header
	result.setHeader(colOffset + n, stats[i].getHeader(n));

	// data
	for (m = 0; m < stats[i].getRowCount(); m++)
	  result.setCell(m, colOffset + n, stats[i].getCell(m, n));
      }

      // inc offset
      colOffset += stats[i].getColumnCount();
    }

    return result;
  }
}
