/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractDataContainerFileWriter.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.lang.reflect.Array;
import java.util.Arrays;

import adams.core.io.PlaceholderDirectory;
import adams.core.io.PlaceholderFile;
import adams.data.DataUtils;
import adams.data.container.DataContainer;
import adams.data.id.DatabaseIDHandler;
import adams.data.io.output.AbstractDataContainerWriter;
import adams.flow.core.Token;

/**
 * Abstract ancestor for transformer actors that write data containers
 * to disk.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @param <T> the type of data to write to disk
 */
public abstract class AbstractDataContainerFileWriter<T extends DataContainer>
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -2589436559371405252L;

  /** the writer to use. */
  protected AbstractDataContainerWriter<T> m_Writer;

  /** the output directory. */
  protected PlaceholderDirectory m_OutputDir;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "writer", "writer",
	    getDefaultWriter());

    m_OptionManager.add(
	    "dir", "outputDir",
	    new PlaceholderDirectory("."));
  }

  /**
   * Returns the default writer to use.
   *
   * @return		the default writer
   */
  protected abstract AbstractDataContainerWriter<T> getDefaultWriter();

  /**
   * Sets the writer to use.
   *
   * @param value	the writer
   */
  public void setWriter(AbstractDataContainerWriter value) {
    m_Writer = value;
    reset();
  }

  /**
   * Returns the writer in use.
   *
   * @return		the writer
   */
  public AbstractDataContainerWriter getWriter() {
    return m_Writer;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String writerTipText() {
    return "The writer to use for saving the data.";
  }

  /**
   * Sets the output directory for the spectrums.
   *
   * @param value	the directory
   */
  public void setOutputDir(PlaceholderDirectory value) {
    m_OutputDir = value;
    reset();
  }

  /**
   * Returns the current output directory for the spectrums.
   *
   * @return		the directory
   */
  public PlaceholderDirectory getOutputDir() {
    return m_OutputDir;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputDirTipText() {
    return "The output directory for the data.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("writer");

    if (variable != null)
      return variable;
    else if (m_Writer != null)
      return m_Writer.getClass().getName();
    else
      return null;
  }

  /**
   * Returns the data container class in use.
   *
   * @return		the container class
   */
  protected abstract Class getDataContainerClass();

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the data type
   */
  public Class[] accepts() {
    Class	cls;

    cls = Array.newInstance(getDataContainerClass(), 0).getClass();

    return new Class[]{cls.getComponentType(), cls};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    PlaceholderFile	file;
    T			cont;
    T[]			conts;
    boolean		success;

    result = null;

    if (m_InputToken.getPayload().getClass().isArray()) {
      conts = (T[]) m_InputToken.getPayload();
      cont  = conts[0];
    }
    else {
      conts = null;
      cont  = (T) m_InputToken.getPayload();
    }

    // setup writer
    if (m_Writer.isOutputFile())
      file = new PlaceholderFile(DataUtils.createFilename(m_OutputDir, (Object) cont, "." + m_Writer.getFormatExtensions()[0]));
    else
      file = new PlaceholderFile(DataUtils.createFilename(m_OutputDir, (Object) cont, null));
    m_Writer.setOutput(file);

    // write data
    try {
      if (conts != null)
	success = m_Writer.write(Arrays.asList(conts));
      else
	success = m_Writer.write(cont);

      if (!success)
	result = "Failed to write data to '" + file + "'!";

      if (isDebugOn()) {
	if (conts != null) {
	  debug(conts.length + " containers written to " + file);
	}
	else {
	  if (cont instanceof DatabaseIDHandler)
	    debug("Container #" + ((DatabaseIDHandler) cont).getDatabaseID() + " written to " + file);
	  else
	    debug("Container " + cont + " written to " + file);
	}
      }
      m_Writer.cleanUp();
      if (success)
	m_OutputToken = new Token(file.getAbsolutePath());
    }
    catch (Exception e) {
      result = "Error writing container(s) to '" + file + "': " + e;
      return result;
    }

    return result;
  }
}
