/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ReportMathExpression.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.core.CloneHandler;
import adams.core.Utils;
import adams.data.report.DataType;
import adams.data.report.Field;
import adams.data.report.MutableReportHandler;
import adams.data.report.Report;
import adams.data.report.ReportHandler;
import adams.flow.core.Token;
import adams.parser.GrammarSupplier;
import adams.parser.MathematicalExpressionText;

/**
 <!-- globalinfo-start -->
 * Evaluates a mathematical expression based on report values.<br/>
 * Either outputs the updated report or the report handler if that allows updating the report. The calculated value can be output by itself, if the 'outputResult' property is enabled.<br/>
 * Variables are supported as well, e.g.: pow(X,&#64;{exp}) with '&#64;{exp}' being a variable available at execution time.<br/>
 * <br/>
 * The following grammar is used for the expressions:<br/>
 * <br/>
 * expr_list ::= expr_list expr_part | expr_part ;<br/>
 * expr_part ::= expr ;<br/>
 * <br/>
 * numexpr   ::=   ( numexpr )<br/>
 *               |  NUMBER<br/>
 *               | -numexpr<br/>
 *               | constexpr<br/>
 *               | opexpr<br/>
 *               | varexpr<br/>
 *               | funcexpr<br/>
 *               ;<br/>
 * <br/>
 * constexpr ::=   PI<br/>
 *               | E<br/>
 *               ;<br/>
 * <br/>
 * opexpr    ::=   numexpr + numexpr<br/>
 *               | numexpr - numexpr<br/>
 *               | numexpr * numexpr<br/>
 *               | numexpr &#47; numexpr<br/>
 *               | numexpr ^ numexpr<br/>
 *               ;<br/>
 * <br/>
 * varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 * funcexpr ::=    abs ( numexpr )<br/>
 *               | sqrt ( numexpr )<br/>
 *               | log ( numexpr )<br/>
 *               | exp ( numexpr )<br/>
 *               | sin ( numexpr )<br/>
 *               | cos ( numexpr )<br/>
 *               | tan ( numexpr )<br/>
 *               | rint ( numexpr )<br/>
 *               | floor ( numexpr )<br/>
 *               | pow ( numexpr , numexpr )<br/>
 *               | ceil ( numexpr )<br/>
 *               | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *               | ifmissing ( variable , numexpr (default value if variable is missing) )<br/>
 *               ;<br/>
 * <br/>
 * boolexpr ::=    BOOLEAN<br/>
 *               | true<br/>
 *               | false<br/>
 *               | istrue ( VARIABLE (must represent a boolean string: true|false) )<br/>
 *               | numexpr &amp;lt; numexpr<br/>
 *               | numexpr &amp;lt;= numexpr<br/>
 *               | numexpr &amp;gt; numexpr<br/>
 *               | numexpr &amp;gt;= numexpr<br/>
 *               | numexpr = numexpr<br/>
 *               | numexpr != numexpr<br/>
 *               | ( boolexpr )<br/>
 *               | ! boolexpr<br/>
 *               | boolexpr &amp; boolexpr<br/>
 *               | boolexpr | boolexpr<br/>
 *               ;<br/>
 * <br/>
 * Variables are either all upper case letters (e.g., "ABC") or any character apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.report.Report<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.report.ReportHandler<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.report.Report<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.report.ReportHandler<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ReportMathExpression
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The mathematical expression to evaluate.
 * &nbsp;&nbsp;&nbsp;default: 42
 * </pre>
 *
 * <pre>-field &lt;adams.data.report.Field&gt; (property: field)
 * &nbsp;&nbsp;&nbsp;The field in the report to update.
 * &nbsp;&nbsp;&nbsp;default: Blah[N]
 * </pre>
 *
 * <pre>-output-result (property: outputResult)
 * &nbsp;&nbsp;&nbsp;If enabled, then the result is output instead of the report field updated.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @see adams.parser.MathematicalExpression
 */
public class ReportMathExpression
  extends AbstractTransformer
  implements GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = -8477454145267616359L;

  /** the mathematical expression to evaluate. */
  protected MathematicalExpressionText m_Expression;

  /** the field for the result. */
  protected Field m_Field;

  /** whether to output the result instead of updating the report. */
  protected boolean m_OutputResult;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Evaluates a mathematical expression based on report values.\n"
      + "Either outputs the updated report or the report handler if that "
      + "allows updating the report. The calculated value can be output by "
      + "itself, if the 'outputResult' property is enabled.\n"
      + "Variables are supported as well, e.g.: pow(X,@{exp}) with '@{exp}' "
      + "being a variable available at execution time.\n\n"
      + "The following grammar is used for the expressions:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new adams.parser.ReportMathExpression().getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "expression", "expression",
	    new MathematicalExpressionText("42"));

    m_OptionManager.add(
	    "field", "field",
	    new Field("Blah", DataType.NUMERIC));

    m_OptionManager.add(
	    "output-result", "outputResult",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("expression");
    if (variable != null)
      result = variable;
    else
      result = m_Expression.stringValue();

    if (!m_OutputResult) {
      result   += ", field: ";
      variable = getOptionManager().getVariableForProperty("field");
      if (variable != null)
	result += variable;
      else
	result += m_Field;
    }

    return result;
  }

  /**
   * Sets the mathematical expression to evaluate.
   *
   * @param value	the expression
   */
  public void setExpression(MathematicalExpressionText value) {
    m_Expression = value;
    reset();
  }

  /**
   * Returns the mathematical expression to evaluate.
   *
   * @return		the expression
   */
  public MathematicalExpressionText getExpression() {
    return m_Expression;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return "The mathematical expression to evaluate.";
  }

  /**
   * Sets the field to update.
   *
   * @param value	the field
   */
  public void setField(Field value) {
    m_Field = value;
    reset();
  }

  /**
   * Returns the field to update.
   *
   * @return		the field
   */
  public Field getField() {
    return m_Field;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String fieldTipText() {
    return "The field in the report to update.";
  }

  /**
   * Sets whether to output the result instead of updating the report field.
   *
   * @param value	true if the result is output instead
   */
  public void setOutputResult(boolean value) {
    m_OutputResult = value;
    reset();
  }

  /**
   * Returns whether to output the result instead of updating the report field.
   *
   * @return		true if the result is output instead
   */
  public boolean getOutputResult() {
    return m_OutputResult;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputResultTipText() {
    return "If enabled, then the result is output instead of the report field updated.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.data.report.Report.class, adams.data.report.ReportHandler.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Report.class, ReportHandler.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.data.report.Report.class, adams.data.report.ReportHandler.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    if (m_OutputResult)
      return new Class[]{Double.class};
    else
      return new Class[]{Report.class, ReportHandler.class};
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if ((m_Expression == null) || (m_Expression.getValue().length() == 0))
	result = "No expression provided!";
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    Report		report;
    String		exp;
    Double		eval;
    ReportHandler	handler;

    result = null;

    exp = m_Expression.getValue();
    try {
      // get input
      if (m_InputToken.getPayload() instanceof ReportHandler) {
	handler = (ReportHandler) m_InputToken.getPayload();
	report  = handler.getReport();
      }
      else {
	handler = null;
	report  = (Report) m_InputToken.getPayload();
      }

      // replace variables with their actual values
      exp = getVariables().expand(exp);

      // evaluate the expression
      eval = adams.parser.ReportMathExpression.evaluate(exp, report);

      if (m_OutputResult) {
	m_OutputToken = new Token(eval);
      }
      else {
	report = report.getClone();
	report.setValue(m_Field, eval);
	if ((handler != null) && (handler instanceof MutableReportHandler)) {
	  if (handler instanceof CloneHandler)
	    handler = (ReportHandler) ((CloneHandler) handler).getClone();
	  else
	    handler = (ReportHandler) Utils.deepCopy(handler);
	  ((MutableReportHandler) handler).setReport(report);
	  m_OutputToken = new Token(handler);
	}
	else {
	  m_OutputToken = new Token(report);
	}
      }
    }
    catch (Exception e) {
      m_OutputToken = null;
      getSystemErr().println("Error evaluating: " + exp);
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
