/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * StringReplace.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.core.Placeholders;
import adams.core.Utils;
import adams.core.base.BaseRegExp;

/**
 <!-- globalinfo-start -->
 * Performs a string replacement, using either String.replaceFirst(...) or String.replaceAll(...). Special characters like \n \r \t and \ need to be escaped properly. The input is expected to be escaped, i.e., the string "\t" will get turned into the character '\t'.<br/>
 * If the 'replace' string contains both, variables and placeholders, then first all variables are expanded and then the placeholders. This ensures that variables containing placeholders expand their placeholders as well. Not expanding placeholders will cause 'Illegal group reference' error messages.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String[]<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String[]<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: StringReplace
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-find &lt;adams.core.base.BaseRegExp&gt; (property: find)
 * &nbsp;&nbsp;&nbsp;The string to find (a regular expression).
 * &nbsp;&nbsp;&nbsp;default: find
 * </pre>
 *
 * <pre>-replace &lt;java.lang.String&gt; (property: replace)
 * &nbsp;&nbsp;&nbsp;The string to replace the occurrences with.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-all (property: replaceAll)
 * &nbsp;&nbsp;&nbsp;If set to true, then all occurrences will be replaced; otherwise only the
 * &nbsp;&nbsp;&nbsp;first.
 * </pre>
 *
 * <pre>-placeholder (property: replaceContainsPlaceholder)
 * &nbsp;&nbsp;&nbsp;Set this to true to enable automatic placeholder expansion for the replacement
 * &nbsp;&nbsp;&nbsp;string.
 * </pre>
 *
 * <pre>-variable (property: replaceContainsVariable)
 * &nbsp;&nbsp;&nbsp;Set this to true to enable automatic variable expansion for the replacement
 * &nbsp;&nbsp;&nbsp;string.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class StringReplace
  extends AbstractStringOperation {

  /** for serialization. */
  private static final long serialVersionUID = -1167336515862285272L;

  /** the key for storing the current actual replace in the backup. */
  public final static String BACKUP_ACTUALREPLACE = "actual replace";

  /** the string to find. */
  protected BaseRegExp m_Find;

  /** the replacement string. */
  protected String m_Replace;

  /** whether to replace all or only the first occurrence. */
  protected boolean m_ReplaceAll;

  /** whether the replace string contains a placeholder, which needs to be
   * expanded first. */
  protected boolean m_ReplaceContainsPlaceholder;

  /** whether the replace string contains a variable, which needs to be
   * expanded first. */
  protected boolean m_ReplaceContainsVariable;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Performs a string replacement, using either String.replaceFirst(...) "
      + "or String.replaceAll(...). Special characters like \\n \\r \\t and \\ "
      + "need to be escaped properly. The input is expected to be escaped, "
      + "i.e., the string \"\\t\" will get turned into the character '\\t'.\n"
      + "If the 'replace' string contains both, variables and placeholders, "
      + "then first all variables are expanded and then the placeholders. This "
      + "ensures that variables containing placeholders expand their placeholders "
      + "as well. Not expanding placeholders will cause 'Illegal group reference' "
      + "error messages.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "find", "find",
	    new BaseRegExp("find"));

    m_OptionManager.add(
	    "replace", "replace",
	    "");

    m_OptionManager.add(
	    "all", "replaceAll",
	    false);

    m_OptionManager.add(
	    "placeholder", "replaceContainsPlaceholder",
	    false);

    m_OptionManager.add(
	    "variable", "replaceContainsVariable",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;
    String	replace;
    String	find;

    variable = getOptionManager().getVariableForProperty("replace");
    if (variable != null)
      replace = variable;
    else
      replace = m_Replace;

    variable = getOptionManager().getVariableForProperty("find");
    if (variable != null)
      find = variable;
    else
      find = m_Find.getValue();

    if (replace.length() > 0)
      result = "replace ";
    else
      result = "remove ";
    result += "'" + find + "'";

    if (replace.length() > 0)
      result += " with '" + replace + "'";

    return result;
  }

  /**
   * Sets the string to find (regular expression).
   *
   * @param value	the string
   */
  public void setFind(BaseRegExp value) {
    m_Find = value;
    reset();
  }

  /**
   * Returns the string to find (regular expression).
   *
   * @return		the string
   */
  public BaseRegExp getFind() {
    return m_Find;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String findTipText() {
    return "The string to find (a regular expression).";
  }

  /**
   * Sets the string to replace the occurrences with.
   *
   * @param value	the string
   */
  public void setReplace(String value) {
    m_Replace = Utils.unbackQuoteChars(value);
    reset();
  }

  /**
   * Returns the string to replace the occurences with.
   *
   * @return		the string
   */
  public String getReplace() {
    return Utils.backQuoteChars(m_Replace);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String replaceTipText() {
    return "The string to replace the occurrences with.";
  }

  /**
   * Sets whether all occurrences are replaced or only the first.
   *
   * @param value	true if all are to be replaced, false if only the first
   */
  public void setReplaceAll(boolean value) {
    m_ReplaceAll = value;
    reset();
  }

  /**
   * Returns whether all occurrences are replaced or only the first one.
   *
   * @return		true if all are to be replaced, false if only the first
   */
  public boolean getReplaceAll() {
    return m_ReplaceAll;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String replaceAllTipText() {
    return "If set to true, then all occurrences will be replaced; otherwise only the first.";
  }

  /**
   * Sets whether the replace string contains a placeholder which needs to be
   * expanded first.
   *
   * @param value	true if replace string contains a placeholder
   */
  public void setReplaceContainsPlaceholder(boolean value) {
    m_ReplaceContainsPlaceholder = value;
    reset();
  }

  /**
   * Returns whether the replace string contains a placeholder which needs to be
   * expanded first.
   *
   * @return		true if replace string contains a placeholder
   */
  public boolean getReplaceContainsPlaceholder() {
    return m_ReplaceContainsPlaceholder;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String replaceContainsPlaceholderTipText() {
    return "Set this to true to enable automatic placeholder expansion for the replacement string.";
  }

  /**
   * Sets whether the replace string contains a variable which needs to be
   * expanded first.
   *
   * @param value	true if replace string contains a variable
   */
  public void setReplaceContainsVariable(boolean value) {
    m_ReplaceContainsVariable = value;
    reset();
  }

  /**
   * Returns whether the replace string contains a variable which needs to be
   * expanded first.
   *
   * @return		true if replace string contains a variable
   */
  public boolean getReplaceContainsVariable() {
    return m_ReplaceContainsVariable;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String replaceContainsVariableTipText() {
    return "Set this to true to enable automatic variable expansion for the replacement string.";
  }

  /**
   * Processes the string.
   *
   * @param s		the string to process
   * @return		the processed string
   */
  protected String process(String s) {
    String	replace;

    // do we need to replace variables?
    replace = m_Replace;
    if (m_ReplaceContainsVariable)
      replace = getVariables().expand(replace);
    if (m_ReplaceContainsPlaceholder)
      replace = Placeholders.getSingleton().expand(replace).replace("\\", "/");

    if (isDebugOn())
      debug("replacement string: " + replace);

    if (m_ReplaceAll)
      return s.replaceAll(m_Find.getValue(), replace);
    else
      return s.replaceFirst(m_Find.getValue(), replace);
  }
}
