/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * BaseTextPane.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 * Copyright (C) JBroFuzz (word wrap)
 * Copyright (C) 2003-2007 Philip Isenhour (setting font)
 */
package adams.gui.core;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dialog.ModalityType;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;

import javax.swing.JInternalFrame;
import javax.swing.JTextPane;
import javax.swing.plaf.ComponentUI;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;

import adams.core.License;
import adams.core.annotation.MixedCopyright;
import adams.gui.chooser.FontChooser;

/**
 * A customized JTextPane. Adds functionality for wordwrap, printing and
 * selecting fonts.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class BaseTextPane
  extends JTextPane {

  /** for serialization. */
  private static final long serialVersionUID = 5053144101104728014L;

  /** whether wordwrap is enabled. */
  protected boolean m_WordWrapEnabled;

  /** the backup of the size. */
  protected Dimension m_SizeBackup;

  /**
   * Creates a new <code>BaseTextPane</code>.  A new instance of
   * <code>StyledEditorKit</code> is
   * created and set, and the document model set to <code>null</code>.
   */
  public BaseTextPane() {
    super();
    initialize();
  }

  /**
   * Creates a new <code>BaseTextPane</code>, with a specified document model.
   * A new instance of <code>javax.swing.text.StyledEditorKit</code>
   *  is created and set.
   *
   * @param doc the document model
   */
  public BaseTextPane(StyledDocument doc) {
    super(doc);
    initialize();
  }

  /**
   * Initializes the member variables.
   */
  protected void initialize() {
    m_WordWrapEnabled = false;
    m_SizeBackup      = getSize();
  }

  /**
   * Tries to determine the frame the container is part of.
   *
   * @return		the parent frame if one exists or null if not
   */
  public Frame getParentFrame() {
    return GUIHelper.getParentFrame(this);
  }

  /**
   * Tries to determine the dialog this panel is part of.
   *
   * @return		the parent dialog if one exists or null if not
   */
  public Dialog getParentDialog() {
    return GUIHelper.getParentDialog(this);
  }

  /**
   * Tries to determine the internal frame this panel is part of.
   *
   * @return		the parent internal frame if one exists or null if not
   */
  public JInternalFrame getParentInternalFrame() {
    return GUIHelper.getParentInternalFrame(this);
  }

  /**
   * Pops up a print dialog.
   */
  public void printText() {
    String 	msg;

    try {
      print(null, null, true, null, null, true);
    }
    catch (Exception ex) {
      ex.printStackTrace();
      msg = "Failed to print:\n" + ex;
      if (getParentDialog() != null)
	GUIHelper.showErrorMessage(getParentDialog(), msg);
      else
	GUIHelper.showErrorMessage(getParentFrame(), msg);
    }
  }

  /**
   * Pops up a dialog for selecting the font.
   */
  public void selectFont() {
    FontChooser 	chooser;

    if (getParentDialog() != null)
      chooser = new FontChooser(getParentDialog());
    else
      chooser = new FontChooser(getParentFrame());
    chooser.setCurrent(getFont());
    chooser.setModalityType(ModalityType.DOCUMENT_MODAL);
    chooser.setVisible(true);
    setTextFont(chooser.getCurrent());
  }

  /**
   * Sets the font of the text pane.
   *
   * @param font	the font to use
   */
  @MixedCopyright(
      author = "Philip Isenhour",
      copyright = "2003-2007 Philip Isenhour",
      license = License.PUBLIC_DOMAIN,
      url = "http://javatechniques.com/blog/setting-jtextpane-font-and-color/"
  )
  public void setTextFont(Font font) {
    // Start with the current input attributes for the JTextPane. This
    // should ensure that we do not wipe out any existing attributes
    // (such as alignment or other paragraph attributes) currently
    // set on the text area.
    MutableAttributeSet attrs = getInputAttributes();

    // Set the font family, size, and style, based on properties of
    // the Font object. Note that JTextPane supports a number of
    // character attributes beyond those supported by the Font class.
    // For example, underline, strike-through, super- and sub-script.
    StyleConstants.setFontFamily(attrs, font.getFamily());
    StyleConstants.setFontSize(attrs, font.getSize());
    StyleConstants.setItalic(attrs, (font.getStyle() & Font.ITALIC) != 0);
    StyleConstants.setBold(attrs, (font.getStyle() & Font.BOLD) != 0);

    // Retrieve the pane's document object
    StyledDocument doc = getStyledDocument();

    // Replace the style for the entire document. We exceed the length
    // of the document by 1 so that text entered at the end of the
    // document uses the attributes.
    doc.setCharacterAttributes(0, doc.getLength() + 1, attrs, false);
  }

  /**
   * Sets the wordwrap state.
   *
   * @param value	whether to wrap or not
   */
  public void setWordWrap(boolean value) {
    if (m_WordWrapEnabled != value) {
      m_WordWrapEnabled = value;

      if (m_WordWrapEnabled)
	setSize(m_SizeBackup);
      else
	setSize(getSize());

      if (getParent() != null) {
	getParent().invalidate();
	getParent().doLayout();
      }
    }
  }

  /**
   * Returns the wordwrap status.
   *
   * @return		true if wordwrap is on
   */
  public boolean getWordWrap() {
    return m_WordWrapEnabled;
  }

  /**
   * Returns true if a viewport should always force the width of this
   * <code>Scrollable</code> to match the width of the viewport.
   *
   * @return 		true if a viewport should force the Scrollables width to
   * 			match its own, false otherwise
   */
  @MixedCopyright(
      copyright = "JBroFuzz",
      license = License.GPL3,
      url = "http://grepcode.com/file/repo1.maven.org/maven2/org.owasp.jbrofuzz/jbrofuzz/2.5/org/owasp/jbrofuzz/util/NonWrappingTextPane.java#NonWrappingTextPane"
  )
  public boolean getScrollableTracksViewportWidth() {
    if (m_WordWrapEnabled) {
      return false;
    }
    else {
      final Component parent = getParent();
      final ComponentUI mComponentUI = getUI();

      if (parent != null)
	return (mComponentUI.getPreferredSize(this).width <= parent.getSize().width);

      return true;
    }
  }
}
